<?php
namespace Website\Widgets\StoreProductGrid;

use Doctrine\ORM\Query\Expr\Join;
use Moto;
use MotoStore\Collection\Collection;
use MotoStore\Collection\Query\Param;
use MotoStore\Content\Content\DI\SMStatic;
use MotoStore\Content\Helper\Link;
use MotoStore\Content\Helper\Pagination;
use MotoStore\Content\Widget\AbstractWidget;
use MotoStore\Order\Cart\Storage;
use MotoStore\Product\Entity\Category;
use MotoStore\Product\Entity\Product;
use MotoStore\Settings\Entity\Currency;
use MotoStore\Settings\Entity\Option;
use Zend\Http\PhpEnvironment\Request;

class Widget extends AbstractWidget
{
    protected $_name = 'store_product_grid';

    /**
     * Products Collection
     * @var array
     */
    protected $products = array ();
    protected $_defaultOptions = array(
        'preset' => 'default',
        'spacing' => array(
            'top' => 'auto',
            'left' => 'auto',
            'right' => 'auto',
            'bottom' => 'auto'
        ),
        'align' => array(
            "desktop" => 'left',
            "tablet" => '',
            "mobile-h" => '',
            "mobile-v" => ''
        ),
        'row_size' => 3,
        'per_page' => 12,
        'thumbnail' => array(
            'hide' => false,
            'zoom' => false,
            'preset' => 'default'
        ),
        'show_toolbar' => true,
        'per_page_label' => 'Items per page:',
        'product_title_style' => '',
        'product_price_style' => '',
        'labelColor' => array(
            'background' => null
        ),
        'labelHexColor' => null,
        'wrapWord' =>  true,
        'button' => array(
            'active' => false,
            'size' => 'medium',
            'preset' => 'default',
            'text' => 'Add to cart'
        ),
        'auto_redirect' => true,
        'use_thumbnails' => false,
);
    /**
     * @var null|Category
     */
    public $category = null;

    /**
     * @var Pagination
     */
    protected $paginator;

    /**
     * @var array
     */
    public $pages = array ();

    /**
     * Items to show
     * @var int
     */
    public $per_page;

    /**
     * Search Keyword
     * @var null
     */
    public $searchKeyword = null;

    /**
     * Order By
     * @var null
     */
    public $order_by= null;

    /**
     * Items to show
     * @var int
     */
    public $limit;

    /**
     * @var []
     */
    public $currency;

    /**
     * Current page
     * @var int
     */
    public $currentPage = 1;
    
    public $totalPages = 0;

    /**
     * @var array
     */
    public $pagination = array ();

    /**
     * @var Request
     */
    protected $request;

    /**
     * @var int
     */
    private $maxPaginatinElements = 5;

    protected $_widgetId = true;

    /**
     * @param array $options
     */
    public function __construct ($options = array())
    {
        $this->request = new Request ();
        $this->category = $this->getCategory (Link::getEntityUri (Link::URI_SECTION_CATEGORY));
        $this->limitSwitchListener($options);
        $this->currencySwitchListener ();
        $this->orderBySwitchListener ();
        $this->filterListener ($options);
        parent::__construct ($options);
    }

    /**
     * @return array
     */
    public function getRenderVariables()
    {
        $storage = new Storage ();


        $this->limit = $storage->getLimit($this->category);
        if (!$this->limit || $this->limit == 0){
            $this->limit = $this->_options['per_page'];
            $storage->setLimit($this->_options['per_page'],$this->category);
        }

        $this->getProductsCollection ();
        $this->createPaginationArray ();

        $this->currentPage = $this->getPagination ()->getPage ();
        $this->totalPages  = $this->getPagination ()->getPagesCount ();
        if ($this->totalPages)
        {
            $this->pages    = range ($this->calculatePaginationRangeStart($this->currentPage), $this->calculatePaginationRangeEnd($this->currentPage));
        }

        $this->currency = $this->getDisplayCurrency ();


        $orderby = $storage->getOrderBy() ;
        $this->order_by = $orderby;
        if(!$this->order_by){
            $this->order_by = isset($this->_options['order_by_default'])?$this->_options['order_by_default']:false;
            $storage->setOrderBy($this->order_by);
        }
        $this->products =  $this->getProductsCollection ();
        $fixedProducts = array();
        $customerDiscount = $this->getCustomerDiscount();
        if($customerDiscount){
            foreach ($this->products as $product){
                $product['price'] = $product['price'] - $product['price']*$customerDiscount/100;
                $fixedProducts [] = $product;
            }
        }
        if ($this->_options['use_thumbnails']){
            $this->applyThumbnails ();
        }

        if (count($fixedProducts)>0){
            $this->products = $fixedProducts;
        }
        $this->wrapWord = isset ($this->_options ['wrapWord'])
            ? $this->_options ['wrapWord']
            : true;
        if($this->isCatalogModeEnabled()){
            $this->properties['hide_prices'] = true;
        }
        return array (
            'currentWidget'        => $this,
            'currencies'    => $this->getCurrencyList (),
            'products'      => $this->products,
            'currentPage'   => $this->getPagination ()->getPage (),
        );
    }

    public function applyThumbnails()
    {
        $tmpProducts = $this->products;
        foreach ($tmpProducts as $key => $product) {
            $this->products[$key]['image'] = $this->getThumbnail($product['image'], $this->_options['tumbnail_size']);
        }

    }
    public function isCatalogModeEnabled(){
        $optRepository = clone SMStatic::getInstance()
            ->get ('MotoStore\EntityManager')
            ->getRepository ('MotoStore\Settings\Entity\Option');

        $catalog_mode = $optRepository ->getOption(Option::OPTION_STORE_SETTINGS_PRODUCT, 'catalog_mode');
        return !!$catalog_mode;
    }

    public  function getThumbnail($image,$thumbnail_size){
        $imageThubmnails = json_decode($image['thumbnails'],true);
        if (isset($imageThubmnails[$thumbnail_size])){
            $pathInfo = pathinfo($image['path']);
            $thumbnailPath  = $pathInfo['dirname'] .
                '/thumbnails/' .
                $pathInfo['filename'] .
                '_' .
                $thumbnail_size .
                '_' .
                $imageThubmnails[$thumbnail_size]['width'] .
                'x' .
                $imageThubmnails[$thumbnail_size]['height'].
                '.' .
                $pathInfo['extension'];
            $filepath = Moto\System::getUploadAbsolutePath($thumbnailPath);
            if(file_exists($filepath)){
                $image['path'] = $thumbnailPath;
            }
        }
        return $image;
    }

    /**
     * @param string $preset
     * @return string
     */
    public function getTemplatePath ($preset = 'default')
    {
        $template = '@plugins/moto-store-plugin/app/widgets/' . $this->_name . '/presets/default/template.html.twig';
        return $template;
    }

    /**
     * @return mixed
     */
    protected function getProductsCollection ()
    {
        if (!$this->products)
        {
            /** @var Collection $collection */
            $storage = new Storage ();



            if ($this->category)
            {
                $cat_id = $this->category->getId ();
                $filters =  $storage->getFilters($cat_id);
                if ($filters && $filters !== 'false'){

                    $collection = SMStatic::getInstance ()
                        ->get ('store.collection.product_catalog_filters');

                } else {
                    $collection = SMStatic::getInstance ()
                        ->get ('store.collection.product_list');
                }


            } else {
                $collection = SMStatic::getInstance ()
                    ->get ('store.collection.product_list');
            }
            $collection
                ->setFilters ($this->createFilterParams ());



            $collection
                ->setOffset ($this->getPagination ()->getFirstResult ())
                ->setLimit($storage->getLimit($this->category));

            $keyword = $this->request->getQuery('keyword');

            if ($keyword)
            {
                $this->searchKeyword = $keyword;

                $collection
                    ->getQuery ()
                    ->andWhere('(l.name LIKE \'%'.$keyword.'%\' OR l.keywords LIkE \'%'.$keyword.'%\')');
            }



            if ($this->category)
            {
                $currency = $this->getDisplayCurrency ();
                $rate = $currency->getExchangeRate();
                $cat_id = $this->category->getId ();
                $filters =  $storage->getFilters($cat_id);


                if ($filters && $filters !== 'false'){
                    if ($filters["price"] && ($filters['price']['min'] || $filters['price']['max']) ){
                        if (!$filters['price']['min'] ){
                            $filters['price']['min'] = 0;
                        }
                        $collection
                            ->getQuery ()
                            ->andWhere('p.price >= ' . ($filters['price']['min'] / $rate));
                        if ($filters['price']['max']){
                            $collection
                                ->getQuery ()
                                ->andWhere('p.price <= ' . ($filters['price']['max'] / $rate));
                        }
                    }
                    if ($filters["brands"] ){
                        $brands_ids = array_keys($filters["brands"]);
                        $brands_ids_string = implode(',',$brands_ids);
                        $collection
                            ->getQuery ()
                            ->andWhere('b.id in ('.$brands_ids_string .')');
                    }

                }

            }


            $orderby = $storage->getOrderBy() ;
            if(!$orderby){
                $orderby = isset($this->_options['order_by_default'])?$this->_options['order_by_default']:false;
            }
                if($orderby == 'date_asc'){
                    $collection
                        ->getQuery ()
                        ->orderBy ('p.created_date', 'ASC');
                }
                elseif($orderby == 'price_desc')
                {
                    $collection
                        ->getQuery ()
                        ->orderBy ('p.price', 'DESC');
                }
                elseif($orderby == 'price_asc')
                {
                    $collection
                        ->getQuery ()
                        ->orderBy ('p.price', 'ASC');
                }
                elseif($orderby == 'name_desc')
                {
                    $collection
                        ->getQuery ()
                        ->orderBy ('l.name', 'DESC');
                }
                elseif($orderby == 'name_asc')
                {
                    $collection
                        ->getQuery ()
                        ->orderBy ('l.name', 'ASC');
                }
                else
                {
                    $collection
                        ->getQuery ()
                        ->orderBy ('p.created_date', 'DESC');

                }


            if ($this->category)
            {
                $cat_id = $this->category->getId ();
                $filters =  $storage->getFilters($cat_id);
                if ($filters && $filters !== 'false' && $filters["properties"]){
                    $collection
                        ->setOffset (0)
                        ->setLimit(9999999);
                }
            }



            $result = $collection
                ->getCollectionWithMetaData ();

            $this
                ->getPagination ()
                ->setResultCount ($result ['totalCount']);

            $this->products = $result ['collection'];
            if ($this->category)
            {
                $cat_id = $this->category->getId ();
                $filters =  $storage->getFilters($cat_id);
                if ($filters && $filters !== 'false'){
                    if ($filters["properties"] ){
                        foreach ($this->products as $key=>$product){
                            $product_prop_values = array();
                            $need_to_show_product = true ;
                            foreach ($product['properties'] as $prop){
                                array_push($product_prop_values,$prop['value_id']);
                            }
                            foreach ($filters["properties"] as $value_arr){
                                $values = array_keys($value_arr);
                                if (count (array_intersect ($values, $product_prop_values)) >0){
                                    $need_to_show_product = ($need_to_show_product && true);
                                } else {
                                    $need_to_show_product = ($need_to_show_product && false);
                                }
                            }
                            if (!$need_to_show_product){
                                unset($this->products[$key]);
                            }
                        }
                        $this
                            ->getPagination ()
                            ->setResultCount (count($this->products));

                        $this->tmp_products = array_slice($this->products, $this->getPagination ()->getFirstResult (), $storage->getLimit($this->category));
                        $this->products = $this->tmp_products;
                    }
                }
            }

        }
        return $this->products;
    }

    /**
     * Create filters array for products collection
     *
     * @return array
     */

    protected function createFilterParams ()
    {
        /** Display only visible */
        $filters = array (
           new Param ('visibility', Param::OPERATOR_EQ, Product::PRODUCT_VISIBLE),
        );

        if(!$this->isCustomerBtob()) {
            $filters [] = new Param ('btob', Param::OPERATOR_NOT_EQ, 1);
        }
        /** If defined category load from category */
        if ($this->category)
        {
            $filters [] = new Param ('category_id', Param::OPERATOR_EQ, $this->category->getId ());

        }


        /** Check out of stock */
        if (!$this->getOptionsRepository ()->getOption (Option::OPTION_STORE_SETTINGS_PRODUCT, 'display_out_stock_product', 0))
        {
            $filters [] = new Param ('quantity', Param::OPERATOR_GREATER, 0);
        }
        return $filters;
    }


    /**
     * @return Currency
     */
    protected function getDisplayCurrency ()
    {
        return SMStatic::getInstance ()
            ->get ('MotoStore\EntityManager')
            ->getRepository ('MotoStore\Settings\Entity\Currency')
            ->getDisplayCurrency ();
    }


    protected  function isCustomerBtob()
    {
        $identity  = SMStatic::getInstance ()
            ->get ('store.authentication.service')
            ->getIdentity ();
        if($identity)
        {
            $customer = SMStatic::getInstance ()
                ->get ('MotoStore\EntityManager')
                ->getRepository ('MotoStore\Customer\Entity\Customer')
                ->find ($identity->getId ());
            if ($customer){
                return (bool)$customer->getBtob ();
            }
        }
        return false;

    }

    protected  function getCustomerDiscount()
    {
        $identity  = SMStatic::getInstance ()
            ->get ('store.authentication.service')
            ->getIdentity ();
        if($identity)
        {
            $customer = SMStatic::getInstance ()
                ->get ('MotoStore\EntityManager')
                ->getRepository ('MotoStore\Customer\Entity\Customer')
                ->find ($identity->getId ());
            if ($customer){
                $discount = $customer->getLoyaltyDiscount();
                if($discount && $discount > 0 && $discount < 100){
                    return $discount;
                } else {
                    return false;
                }
            }
        }
        return false;

    }

    /**
     * Get first element of pages range
     *
     * @param $currentPage
     * @return int
     */
    private function calculatePaginationRangeStart ($currentPage)
    {
        $position = ceil($currentPage / $this->maxPaginatinElements) - 1;

        if ($position == 0)
        {
            return 1;
        }
        return $position * $this->maxPaginatinElements;
    }

    /**
     * Get last element of pages range
     * @param $currentPage
     * @return int
     */
    private function calculatePaginationRangeEnd ($currentPage)
    {
        $start = $this->calculatePaginationRangeStart ($currentPage) + $this->maxPaginatinElements;
        $total = $this->getPagination ()->getPagesCount ();

        if ($start >= $total)
        {
            $start = $total;
        }
        return $start;
    }


    /**
     * Create Pagination Values
     *
     * @return array
     */
    protected function createPaginationArray ()
    {
        $limit = $this->_options['per_page'];
        for ($i=1; $i <= 3; ++$i)
        {
            $this->pagination [] = $limit * pow($i,2);
        }
    }


    /**
     * @return Pagination
     */
    protected function getPagination ()
    {
        if (!$this->paginator)
        {
            $this->paginator = new Pagination ($this->request);
            $storage = new Storage ();
            $limit = $storage->getLimit($this->category)? $storage->getLimit($this->category): $this->_options['per_page'];
            $this->paginator->setLimit ($limit);
        }
        return $this->paginator;
    }

    protected function getCurrencyList ()
    {
        return SMStatic::getInstance ()
            ->get ('MotoStore\EntityManager')
            ->getRepository ('MotoStore\Settings\Entity\Currency')
            ->getActiveCurrenciesList ();
    }

    /**
     * @param $uri
     * @return mixed
     */
    protected function getCategory ($uri)
    {
        return SMStatic::getInstance ()
            ->get ('MotoStore\EntityManager')
            ->getRepository ('MotoStore\Product\Entity\Category')
            ->getCategory ($uri);
    }

    /**
     * Currency Listener
     */
    protected function currencySwitchListener ()
    {
        $currency = $this->request->getPost ('currency_id');


        if ($currency)
        {
            $storage = new Storage ();
            $storage->setCurrency ($currency);
            header ('Location: ' . $this->request->getRequestUri ());
            exit;
        }

    }
    /**
     * OrderBy Listener
     */
    protected function orderBySwitchListener ()
    {
        $order_by = $this->request->getPost ('order_by');

        if ($order_by)
        {

            $storage = new Storage ();
            $storage->setOrderBy ($order_by);
            header ('Location: ' . $this->request->getRequestUri ());
            exit;
        }

    }
    /**
     * Limit Listener
     */
    protected function limitSwitchListener ($options)
    {

        $storage = new Storage ();
        $currlimit = $storage->getLimit();

        $limit = $this->request->getQuery ('limit', 0);
        if ($limit && is_numeric ($limit) && $limit > 0)
        {
            $storage = new Storage ();
            $storage->setLimit($limit,$this->category);
        } else if (is_numeric ($limit) && $limit = 0 && $currlimit == null){
            $storage = new Storage ();
            $storage->setLimit($options ['properties']['per_page'],$this->category);
        }

        return true;

    }
    protected function filterListener ($options)
    {

        if ($this->category)
        {
            $storage = new Storage ();

            $cat_id = $this->category->getId ();
            $filters = $this->request->getQuery ('filters', null);
            if ($filters){
                $storage->setFilters($filters, $cat_id);
            }

        }

        return true;

    }
}