<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com>
 **********************************************************************************************************************/

namespace MotoStore\Collection;


use Doctrine\ORM\EntityManager;

abstract  class AbstractImport
{
    /**
     * Count of updated entities
     * @var int
     */
    private $updatedCounter = 0;

    /**
     * Count of inserted elements
     * @var int
     */
    private $insertedCounter = 0;

    /**
     * Count of filed entities
     * @var int
     */
    private $failedCounter = 0;

    /**
     * @var EntityManager
     */
    protected $entityManager;

    /** @var resource  */
    private $resource;

    /**
     * Headers Titles Indexes from csv
     * @var array
     */
    private $headersIndexes = array ();

    /**
     * Valid Headers Aliases
     * which are sorted in order such as indexes
     * @var array
     */
    private $headers = array ();

    /**
     * AbstractImport constructor.
     * @param EntityManager $entityManager
     */
    public function __construct (EntityManager $entityManager)
    {
        $this->entityManager = $entityManager;
    }

    /**
     * Get allowed CSV headers
     * @return array
     */
    protected abstract function getHeaders ();

    /**
     * Retrieve Entity to Update
     *
     * @param array $data
     * @return \stdClass|null
     */
    protected abstract function getUpdateEntity ($data = array ());

    /**
     * Handle new data to update entity
     *
     * @param $entity
     * @param $data
     * @return bool
     */
    protected abstract function handle ($entity, $data);

    /**
     * Return the prototype of import entity
     *
     * @return \stdClass
     */
    protected abstract function getImportEntityPrototype ();

    /**
     * Transform data to form type array
     *
     * @param array $data
     * @return mixed
     */
    protected abstract function transform ($data = array ());


    protected function preimportProcess ()
    {

    }

    protected function postimportProcess ()
    {

    }


    public function import ($resource)
    {

        if (!is_resource ($resource))
        {
            error_log ('Bad resource for import ' . get_class ($this->getImportEntityPrototype ()));

            return;
        }
        //error_log('in_import');
        $this->resource = $resource;
        $this->preimportProcess ();
        //error_log(json_encode($resource));
        while ($data = fgetcsv($this->resource))
        {
            //error_log(json_encode($data));
            // Export first row as headers
            if (empty ($this->headers))
            {
                $this->extractHeaders ($data);
                continue;
            }
            //error_log(json_encode($data));
            // Extract fields with exists headers indexes
            $extractedData  = array_intersect_key ($data, $this->headersIndexes);
            // Combine with allowed headers
            $combined = array_combine ($this->headers, $extractedData);
            // proceed row data
            //error_log(json_encode($combined));
            $this->proceedData ($combined);
        }

        $this->postimportProcess ();
    }



    /**
     * Remove All Invalid Headers
     * @param $data
     */
    protected function extractHeaders ($data)
    {
        $validHeaders = array_intersect_key (array_flip ($data), array_flip ($this->getHeaders ()));

        //error_log(json_encode($validHeaders));
        $this->headersIndexes = array_flip ($validHeaders);
        $this->headers = array_keys ($validHeaders);
    }

    /**
     * Proceed data import
     *
     * @param $data
     */
    protected function proceedData ($data)
    {
        // Flag for update session
        $isUpdate = false;

        $entity = $this->getUpdateEntity ($data);

        empty ($entity)
            ? $entity = $this->getImportEntityPrototype ()
            : $isUpdate = true;

        try
        {
            $isError = !$this->handle ($entity, $this->transform ($data));

            $this->updateCounters ($isUpdate, $isError);
        }
        catch (\Exception $e)
        {
            $this->updateCounters ($isUpdate, true);

            error_log (sprintf ('Store import error %s: %s ', get_class ($this->getImportEntityPrototype ()), $e->getMessage ()));
            error_log ('Store Import Data : ' . var_export ($data, true));
        }

    }

    /**
     * Update import session counters
     *
     * @param $isUpdate
     * @param $isError
     */
    private function updateCounters ($isUpdate, $isError = false)
    {
        if ($isError)
        {
            ++$this->failedCounter;

            return;
        }

        $isUpdate
            ? ++$this->updatedCounter
            : ++$this->insertedCounter;
    }

    public function getResult ()
    {
        return array (
            'updated'   => $this->updatedCounter,
            'inserted'  => $this->insertedCounter,
            'failed'    => $this->failedCounter,

        );
    }
}