<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Collection\Query;


use Doctrine\ORM\QueryBuilder;

class Designer
{
    /**
     * Query Filter Prams
     * @var array
     */
    protected $params = array ();

    protected $metaParam;

    /**
     * map query to entity
     * @var array
     */
    protected $propertiesMap;

    /**
     * @var QueryBuilder
     */
    protected $query;

    /**
     * @param $propertiesMap
     * @param $query
     */
    function __construct ($query, $propertiesMap)
    {
        $this->query = $query;
        $this->propertiesMap = $propertiesMap;
    }

    /**
     * @param $name
     * @param $operation
     * @param $value
     */
    public function addFilter ($name, $operation, $value)
    {
        $this->params [] = new Param ($name, $operation, $value);
    }

    /**
     * @param Param $param
     */
    public function addParam (Param $param)
    {
        $this->params [] = $param;
    }

    /**
     * Reset Query Params
     */
    public function resetParams ()
    {
        $this->params = array ();
    }



    /**
     * Get mapped entity field
     *
     * @param string $field
     * @return string
     * @throws QueryException
     */
    protected function getPropertyName ($field)
    {
        if (array_key_exists ($field, $this->propertiesMap))
        {
            return $this->propertiesMap [$field];
        }

        throw new QueryException ('Invalid alias for property');
    }

    /**
     * Apply filter params
     */
    public function applyParams ()
    {
        $this
            ->query
            ->resetDQLPart('where');

        array_map (array ($this, 'applyParamExpression'), $this->params);
    }

    /**
     * Apply order params
     *
     * @param $order
     * @param $direction
     * @throws QueryException
     */
    public function addOrderParams ($order, $direction)
    {
        $this->query->addOrderBy ($this->getPropertyName ($order), $direction);
    }

    /**
     * Reset Order
     */
    public function resetOrderParams ()
    {
        $this->query->resetDQLPart ('orderBy');
    }

    /**
     * Apply param to Query
     * @param Param $param
     * @throws QueryException
     */
    protected function applyParamExpression (Param $param)
    {
        $property = $this->getPropertyName($param->getField());

        switch ($param->getOperator()) {
            case Param::OPERATOR_EQ:

                $this->query->andWhere($this->query->expr()->eq($property, $this->query->expr()->literal($param->getValue())));
                break;

            case Param::OPERATOR_NOT_EQ:

                $this->query->andWhere($this->query->expr()->neq($property, $this->query->expr()->literal($param->getValue())));
                break;

            case Param::OPERATOR_GREATER:

                $this->query->andWhere($this->query->expr()->gt($property, $this->query->expr()->literal($param->getValue())));
                break;

            case Param::OPERATOR_LESS:

                $this->query->andWhere($this->query->expr()->lt($property, $this->query->expr()->literal($param->getValue())));
                break;

            case Param::OPERATOR_START:

                $this->query
                    ->andWhere("{$property} LIKE :{$param->getField()}")
                    ->setParameter($param->getField(), $param->getValue() . '%');
                break;

            case Param::OPERATOR_CONTAINS:

                $this->query
                    ->andWhere("{$property} LIKE :{$param->getField()}")
                    ->setParameter($param->getField(), '%' . $param->getValue() . '%');
                break;

            case Param::OPERATOR_FINISH:

                $this->query
                    ->andWhere("{$property} LIKE :{$param->getField()}")
                    ->setParameter($param->getField(), '%' . $param->getValue());
                break;

            case Param::OPERATOR_NULL:

                $this->query->andWhere($this->query->expr()->isNull($property));
                break;

            case Param::OPERATOR_IN:

                $this->query->andWhere($this->query->expr()->in($property, $param->getValue()));
                break;

            case Param::OPERATOR_BETWEEN:
                $this->query->andWhere($this->query->expr()->gt($property, $this->query->expr()->literal($param->getValue()[0])));
                $this->query->andWhere($this->query->expr()->lt($property, $this->query->expr()->literal($param->getValue()[1])));
                break;

            default:
                break;
        }
    }
}