<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Content\Application;

use Felipecwb\Routing\Exception\RouteNotFoundException;
use Moto\Application\Pages\PageModel;
use MotoStore\Content\Content\DI\SMStatic;
use MotoStore\Content\Plugin\Metatag;
use MotoStore\Content\Repository\PageRepository;
use MotoStore\Product\Entity\Category;
use MotoStore\Product\Entity\Product;
use MotoStore\Settings\Entity\Option;
use MotoStore\Settings\Repository\OptionRepository;
use Zend\ServiceManager\ServiceLocatorInterface;

class StoreApplication extends AbstractApplication
{
    /** @var  Metatag */
    private  $metatag;

    /**
     * @var PageRepository
     */
    private  $pageRepository;

    /**
     * @var OptionRepository
     */
    private $optionRepository;

    /**
     * StoreApplication constructor.
     */
    public function __construct ()
    {
        $this->optionRepository = $this
            ->getServiceLocator ()
            ->get ('MotoStore\EntityManager')
            ->getRepository ('MotoStore\Settings\Entity\Option');

        $this->metatag = $this
            ->getServiceLocator ()
            ->get ('MotoStore\Plugin\Metatag');

        $this->pageRepository = $this
            ->getServiceLocator ()
            ->get ('MotoStore\Content\PageRepository');

        parent::__construct ();
    }

    /**
     * Is home page flag
     * @var bool
     */
    private static $isHome;

    /**
     * Page provider
     * @param null $url
     * @return \Moto\Application\Pages\PageModel
     */
    public function getPage ($url = null)
    {
        try
        {
            $user = \Moto\System::getUser();
            if (!$this->isMaintenanceMode () || $user)
            {

                $user = \Moto\System::getUser();
                $isPreview = $this->isPreviewMode();
                
                $authentication = SMStatic::getInstance()->get ('store.authentication.service');

                $this
                    ->getRenderEngine ()
                    ->addGlobal ('account', $authentication->getIdentity ());

                $this
                    ->registerStoreRoutes ();

                $toDispatch = self::isHome ()
                    ? '/store'
                    : $this->getRequestUrl ();

                $content = $this
                    ->getRouter ()
                    ->dispatch ($toDispatch);

                $protectedPageId = \Moto\Website\Settings::get('password_protected_page_id');

                if ($content && $content->id == $protectedPageId && !$user)
                {
                    $content = null;
                }

                if ($content && !$isPreview && !$content->isAllowViewContent($user)) {
                    $protectedPage = $content;

                    if ($protectedPageId)
                    {
                        $content = \Moto\Application\Pages\Service::getById($protectedPageId);
                        if ($content)
                        {
                            $content->title = $protectedPage->title;
                            $this->getRenderEngine()->addGlobal('protectedPage', $protectedPage);
                            $this->getResponse()->setStatusCode(403);
                        }
                    } else
                    {
                        $content = null;
                    }
                }

                if($content && !$content->relative_url && ( $content->type == 'template.store.product' ||  $content->type == 'template.store.category' ) && !isset($_REQUEST['keyword']))
                {
                    return $this->getNotFoundPage ();
                }

                if ($content)
                {
                    $this
                        ->metatag
                        ->applyPageMetadata ($content);

                    $this
                        ->metatag->attachMetaOgEvent ($content);

                    return $content;
                }



            } else {
                return $this->getUnderConstructionPage ();
            }

            return $this->getNotFoundPage ();

        }
        catch (RouteNotFoundException $e)
        {
            return $this->getNotFoundPage ();
        }
    }

    /**
     * Register Store Routes
     */
    protected function registerStoreRoutes ()
    {
        $config = SMStatic::getInstance ()->get ('store.config');

        foreach ($config ['routes'] as $route_name => $route_options)
        {
            $this->addRoute ($route_name, $route_options);
        }
    }

    /**
     * Add sore front rout
     *
     * @param $name
     * @param $options
     */
    protected function addRoute ($name, $options)
    {
        $page_repository = $this->pageRepository;

        $this
            ->getRouter ()
            ->add (
                $options ['pattern'],
                function ($uri = null) use ($page_repository, $options)
                {

                    /** @var PageModel $page */
                    $page = $page_repository->findBySection ($options ['section']);

                    $controller = $options ['controller'];

                    if (class_exists ($options ['controller']))
                    {
                        $controller = new $controller ($page);
                        $execute    = array ($controller, $options ['section']);

                        if (is_callable ($execute))
                        {
                            if (!empty ($page))
                            {
                                call_user_func ($execute, $uri);
                            }
                            else
                            {
                                $page = $page_repository->getEmptyPage ();
                                $page->content = call_user_func ($execute, $uri);
                            }
                            return $page;
                        }
                    }

                    error_log('Route not found:' . var_export ($options, true));
                    throw new RouteNotFoundException;
                }
            );
    }

    public static function isHome ()
    {
        if (!is_null (self::$isHome)) {
            return self::$isHome;
        }
        if (!in_array (self::getRequestUrl (), array ('', '/')))
        {
            self::$isHome = false;
        }

        if (is_null (self::$isHome))
        {
            /** @var OptionRepository $repository */
            $repository = SMStatic::getInstance ()
                ->get ('MotoStore\EntityManager')
                ->getRepository ('MotoStore\Settings\Entity\Option');

            self::$isHome = $repository->getOption (Option::OPTION_STORE_SETTINGS_PRODUCT, 'is_home', false);

        }

        return self::$isHome;
    }

    /**
     * Check for store maintenance mode
     * @return mixed
     */
    public function isMaintenanceMode ()
    {
        $maintenance = $this->optionRepository
            ->getOption (Option::OPTION_STORE_SETTINGS_PRODUCT, 'maintenance', false);

        return $maintenance;
    }

    /**
     * @return ServiceLocatorInterface
     */
    public function getServiceLocator ()
    {
        return SMStatic::getInstance ();
    }

    /**
     * @inheritdoc
     */
    public function bootAsHomePage()
    {
        static::$isHome = true;
    }

    /**
     * @inheritdoc
     */
    public function isMainPage($page)
    {
        return $this->pageRepository->isMainPage($page);
    }

    /**
     * @inheritdoc
     */
    public function getMainPage()
    {
        return $this->pageRepository->findBySection('home');
    }
}