<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Dashboard\Service;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\Query;
use MotoStore\Order\Entity\Order;
use MotoStore\Service\StoreServiceAbstract;
use Zend\ServiceManager\ServiceLocatorInterface;

/**
 * Simple store analytics
 *
 * Class AnalyticsServiceClass
 * @package MotoStore\Analytics\Service
 */
class DashboardServiceClass extends StoreServiceAbstract
{
    /**
     * @var EntityManager
     */
    protected $entityManager;

    protected $_resourcePrivilegesMap = array (
        'getConversationsNumbers' => 'get',
        'getSalesStatistic' => 'get',
        'lastOrders' => 'get',
    );

    /**
     * @param ServiceLocatorInterface $serviceLocator
     */
    public function __construct (ServiceLocatorInterface $serviceLocator)
    {
        $this->applyResourcePrefix ();
        $this->entityManager = $serviceLocator->get ('MotoStore\EntityManager');
    }

    /**
     * Prefix of Resource
     *
     * @return mixed
     */
    protected function getPrefix ()
    {
        return 'dashboard';
    }

    /**
     * Collection Name
     *
     * @return string
     */
    protected function getCollectionName () {}

    protected function createQuery ()
    {
        return $this
            ->entityManager
            ->createQueryBuilder ();
    }

    public function getConversationsNumbers ($rangeValue)
    {
        $rangetext = '-9999 days';
        if($rangeValue == 'Last Day'){
            $rangetext = '-1 days';
        }
        if($rangeValue == 'Last Week'){
            $rangetext = '-7 days';
        }
        if($rangeValue == 'Last Month'){
            $rangetext = '-30 days';
        }

        $query = $this->createQuery ();

        $query
            ->select ('count(o.id)')
            ->from ('MotoStore\Order\Entity\Order','o')
            ->where ($query->expr ()->gt ('o.created_at', '?1'))
            ->setParameter (1, new \DateTime ($rangetext));

        $orders = $query
            ->getQuery ()
            ->getSingleScalarResult ();

        $query = $this->createQuery ();

        $query
            ->select ('count(c.id)')
            ->from ('MotoStore\Customer\Entity\Customer','c')
            ->where ($query->expr ()->gt ('c.created_at', '?1'))
            ->setParameter (1, new \DateTime ($rangetext) );

        $customers = $query
            ->getQuery ()
            ->getSingleScalarResult ();

        $query = $this->createQuery ();

        $query
            ->select ('count(o.id)')
            ->from ('MotoStore\Order\Entity\Order','o')
            ->where ($query->expr ()->gt ('o.created_at', '?1'))
            ->setParameter (1, new \DateTime ('- 1 days'));

        $dailyOrders = $query
            ->getQuery ()
            ->getSingleScalarResult ();

        $query = $this->createQuery ();

        $query
            ->select ('count(c.id)')
            ->from ('MotoStore\Customer\Entity\Customer','c')
            ->where ($query->expr ()->gt ('c.created_at', '?1'))
            ->setParameter (1, new \DateTime ('- 1 days') );

        $dailyCustomers = $query
            ->getQuery ()
            ->getSingleScalarResult ();

        $query = $this->createQuery ();

        $query
            ->select ('SUM(o.total_price)')
            ->from ('MotoStore\Order\Entity\Order','o')
            ->where ($query->expr ()->gt ('o.created_at', '?1'))
            ->setParameter (1, new \DateTime ($rangetext) )
            ->andWhere ($query->expr ()->in ('o.status', array (Order::ORDER_STATUS_COMPLETE, Order::ORDER_STATUS_SHIPPED, Order::ORDER_PAYMENT_ACCEPTED)));




        $lifeTimeSales = $query
            ->getQuery ()
            ->getSingleScalarResult ();

        $query = $this->createQuery ();

        $query
            ->select ('count(p.id)')
            ->from ('MotoStore\Product\Entity\Product','p');

        $products = $query
            ->getQuery ()
            ->getSingleScalarResult ();

        return array (
            'customers'         => $customers,
            'orders'            => $orders,
            'products'          => $products,
            'daily_customers'   => $dailyCustomers,
            'daily_orders'      => $dailyOrders,
            'life_time_sales'   => $lifeTimeSales ? $lifeTimeSales : 0.00,
        );

    }


    /**
     * @return array
     */
    public function getSalesStatistic ($rangeValue)
    {
        $rangetext = '-999 days';
        $interval = 999;
        if($rangeValue == 'Last Day'){
            $rangetext = '-1 days';
            $interval = 1;
        }
        if($rangeValue == 'Last Week'){
            $rangetext = '-7 days';
            $interval = 7;
        }
        if($rangeValue == 'Last Month'){
            $rangetext = '-30 days';
            $interval = 30;
        }
        $customerQuery = $this
            ->entityManager
            ->createQueryBuilder ();

        $orderQuery = $this
            ->entityManager
            ->createQueryBuilder ();

        $orders = $orderQuery->select ('o.id , o.created_at')
            ->from ('MotoStore\Order\Entity\Order', 'o')
            ->where ($orderQuery->expr ()->gt ('o.created_at', '?1'))
            ->orderBy ('o.created_at', 'desc')
            ->setParameter (1, new \DateTime ($rangetext) );

        $customers = $customerQuery->select ('c.id , c.created_at')
            ->from ('MotoStore\Customer\Entity\Customer', 'c')
            ->where ($customerQuery->expr ()->gt ('c.created_at', '?1'))
            ->orderBy ('c.created_at', 'desc')
            ->setParameter (1, new \DateTime ($rangetext) );

        $orderResult = $orders
            ->getQuery ()
            ->getArrayResult ();

        $customerResult = $customers
            ->getQuery ()
            ->getArrayResult ();

        $now = new \DateTime ('midnight + 1 days');

        $intervals = array ();
        $customerIntervals = array ();
       // var_dump($orderResult);
       // var_dump($customerResult);


        foreach ($orderResult as $order)
        {
            $intervals [] = $now->diff ($order['created_at'])->format('%a');
        }



        foreach ($customerResult as $customer)
        {
            $customerIntervals [] = $now->diff ($customer['created_at'])->format('%a');
        }


        $countedIntervals = array_count_values ($intervals);
        $countedCustomerIntervals = array_count_values ($customerIntervals);

        $sales        = array ();
        $accounts     = array ();
        $categories   = array ();


        for ($dayInterval = $interval; $dayInterval >= 0; --$dayInterval)
        {
            $statisticDay = new \DateTime ("- $dayInterval days");

            $salesCount = 0;
            $accountsCount = 0;

            if (array_key_exists ($dayInterval, $countedIntervals))
            {
                $salesCount = $countedIntervals[$dayInterval];
            }

            if (array_key_exists ($dayInterval, $countedCustomerIntervals))
            {
                $accountsCount = $countedCustomerIntervals[$dayInterval];
            }

            $categories []  = $statisticDay->format ('Y-m-d');
            $sales []       = $salesCount;
            $accounts []    = $accountsCount;

        }

        return array(
            'categories'    => $categories,
            'sales'         => $sales,
            'customers'     => $accounts,
        );
    }

    public function lastOrders ()
    {
        $orders = $this
            ->entityManager
            ->createQueryBuilder ()
            ->select (array ('o'))
            ->from ('MotoStore\Order\Entity\Order', 'o')
            ->setMaxResults (10)
            ->orderBy ('o.created_at', 'DESC')
            ->getQuery()
            ->getArrayResult ();

       // var_dump($orders);

        return array_map (function ($order) {

            $order ['status_name'] = Order::getStatusName($order ['status']);



            return $order;

        }, $orders);
    }
}