<?php


namespace MotoStore\Order\Service;


use MotoStore\Collection\AbstractExport;
use MotoStore\Collection\Collection;
use MotoStore\Collection\ExportInterface;
use MotoStore\Content\Content\DI\SMStatic;
use MotoStore\Order\Entity\Order;

class OrderExportCommon extends AbstractExport implements ExportInterface
{
    /**
     * CSV delimiter
     * @var string
     */
    private $delimiter = ';';

    /**
     * CSV enclosure
     * @var string
     */
    private $enclosure = '"';


    /**
     * Collection to Export
     * @var Collection
     */
    private $collection;

    private $dates;

    /**
     * Target resource
     * @var
     */
    private $resource;

    private $categories = array ();

    /**
     * AbstractExport constructor.
     * @param $target
     * @param Collection $collection
     */
    public function __construct ($target, Collection $collection, $dates = null)
    {
        $this->target = $target;
        $this->collection = $collection;
        $this->dates = $dates;
    }


    /**
     * CSV columns
     * @return array
     */
    public function getHeaders ()
    {
        return array (
            'Order number',
            'Customer number',
            'Date of the order',
            'Total price',
            'Status ID',
            'First Name',
            'Last Name',
            'Company',
            'Street',
            'Postcode',
            'Country',
            'E-mail',
            'Phone',
        );

    }

    /**
     * Prepare record before put to csv
     *
     * @param $record
     * @return array
     */
    public function prepareRecord ($record)
    {
        $record['Order number'] = $record['order_id'];
        $record['Status ID'] = $record['status'];
        $record['Customer number'] = $record['customer_id'];
        $createdDate   = $record ['created_at'];
        $record ['Date of the order'] = $createdDate ? $createdDate->format ('d.m.Y') : '';
        $record ['Company'] = $record ['company'];
        $record ['Last Name'] = $record['last_name'];
        $record ['First Name'] = $record['first_name'];
        $record ['Street'] = $record['address1'];
        $record ['Postcode'] = $record['post_code'];
        $country  = $record['country'];
        $record ['Country'] = $country['iso_3'];
        $record ['E-mail'] = $record['email'];
        $record ['Phone'] = $record['contact_phone'];
        $record ['Total price'] = $record['total_price'];

        return $record;
    }



    public function export ()
    {
        $date_from = date("Ymd", strtotime($this->dates['from']));
        $date_to = date("Ymd", strtotime($this->dates['to']));
        $this->open ();
        $records_without_ids = array();
        $records = array();
        $tmp_records = $this->collection
            ->getCollection ();
        foreach ($tmp_records as $record){
            $createdDate   = $record ['created_at'];
            $formatted_date =  $createdDate ? $createdDate->format ('Ymd') : '';
            //error_log(json_encode(array($formatted_date, $date_from, $date_to)));
            if ($formatted_date >= $date_from && $formatted_date <= $date_to){
                $records_without_ids[] = $this->prepareRecord($record);
            }

        }
        foreach ($records_without_ids as $key=>$record){
            $record['id'] = $key;
            $records[] = $record;
        }

        $this->write ($this->getHeaders ());

        foreach ($records as $record)
        {
            $this->write ($this->excludeFields ($record));
        }
        $this->close ();
    }

    /**
     * Filter record by headers
     * @param $record
     * @return array
     */
    private function excludeFields ($record)
    {
        $headers = array_flip ($this->getHeaders ());

        return array_intersect_key (array_merge ($headers, $record), $headers);
    }

    /**
     * Create Target Resource
     */
    private function open ()
    {
        $this->resource = fopen ($this->target, 'w');
    }

    /**
     * Write record to CSV
     * @param array $record
     */
    private function write (array $record)
    {
        fputcsv ($this->resource, $record, $this->delimiter, $this->enclosure);
    }

    /**
     * Close resource
     */
    private function close ()
    {
        return fclose ($this->resource);
    }


}