<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Order\Service;

    use Doctrine\ORM\Query;
    use Moto\Json\Server\Exception;
    use MotoStore\Content\Cart\Storage;
    use MotoStore\Order\Entity\Comment;
    use MotoStore\Order\Entity\Order;
    use MotoStore\Order\Form\CommentType;
    use MotoStore\Order\Form\OrderType;
    use MotoStore\Service\ServiceEditableInterface;
    use MotoStore\Service\ServiceReadableInterface;
    use MotoStore\Service\StoreServiceAbstract;
    use MotoStore\Content\Helper\PDFGenerator;
    use MotoStore\Settings\Entity\Option;
    use Moto;

class OrderServiceClass extends StoreServiceAbstract implements ServiceReadableInterface, ServiceEditableInterface
{
    protected $_resourcePrivilegesMap = array (
        'getCollection' => 'get',
        'getStatuses' => 'get',
        'getItem' => 'get',
    );

    /**
     * Prefix of Resource
     *
     * @return mixed
     */
    protected function getPrefix ()
    {
        return 'order';
    }

    /**
     * Collection Name
     *
     * @return string
     */
    protected function getCollectionName ()
    {
        return 'store.collection.order';
    }


    /**
     * Save Resource

     * @return mixed
     */
    public function save ()
    {
        return $this
            ->handleUpdateRequest (new OrderType (), new Order ());
    }

    /**
     * Remove Resource
     *
     * @param $id
     * @return mixed
     */
    public function delete ($id)
    {
        $this->handleDeleteRequest ($id);
    }


    public function generateInvoice ($id)
    {
        return PDFGenerator::generateInvoice($id);
    }
    /**
     * Retrieve Collection By Params
     *
     * @return mixed
     */
    public function getCollection ()
    {
        return $this
            ->handleCollectionRequest (
                function (&$order)
                {
                    $order ['status_name'] = Order::getStatusName ($order ['status']);

                    /** @var \DateTime $createdAt */
                    $createdAt = $order ['created_at'];

                    if ($createdAt) {
                        $order ['created_at'] = $createdAt->getTimestamp() * 1000;
                    }

                    /** @var \DateTime $updatedAt */
                    $updatedAt = $order ['updated_at'];

                    if ($updatedAt) {
                        $order ['updated_at'] = $updatedAt->getTimestamp() * 1000;
                    }

                    return $order;
                }
            );
    }

    /**
     * @param int $id
     * @return mixed
     */
    public function getItem ($id)
    {
       return $this
           ->collection
           ->setHydrationMode (Query::HYDRATE_ARRAY)
           ->getOne ($id,
               function (&$order)
               {
                   $order ['status_name'] = Order::getStatusName ($order ['status']);
               }
           );
    }

    /**
     * @return array
     */
    public function getStatuses ()
    {
        return array (
            'collection' => Order::getOrderStatuses (),
            'totalCount' => count (Order::getOrderStatuses ()),
        );
    }

    /**
     * Update Order Status
     */
    public function updateStatus ()
    {
        $comment    = new Comment ();
        $form       = new CommentType ();

        $form
            ->setEntityManager ($this->collection->getEntityManager ())
            ->init ();

        if ($form->handle ($this->_request->getParams (), $comment))
        {
            $order = $this->getItem ($comment->getOrderId ());
            if ($order ['status'] == Order::ORDER_PAYMENT_ACCEPTED){
                \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_PAID",$order ['order_id']);
            }
            else {
                \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_STATUS_UPDATED",array('order_id' => $order ['order_id'], 'status' => $order ['status']));
            }
            if($order ['status'] == Order::ORDER_STATUS_CANCELED){
                $em = $this
                    ->collection
                    ->getEntityManager ();
                $this->restoreInventory($order ['order_id']);
                $em->flush();
            }
            return $order;
        }

        throw new Exception ('ERROR.BAD_REQUEST', 0, $form->getMessages ());
    }

    /**
     * Update Order Status for Bulk Update
     */
    public function updateStatusBulk ()
    {

        $params = $this->_request->getParams ();

        if (!empty ($params ['order_id']) && !empty($params['status']))
        {
            $order_id     = $params ['order_id'];

            $em = $this
                ->collection
                ->getEntityManager ();
            /** @var Order $entity */
            $entity = $em->getRepository ('MotoStore\Order\Entity\Order')
                ->find ($order_id);
            $entity
                ->setStatus($params['status']);
            if ($params ['status'] == Order::ORDER_PAYMENT_ACCEPTED){
                //legacy hook for "order paid" status that is also used by extensions.
                \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_PAID",$entity->getOrderId());
            } else {
                \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_STATUS_UPDATED",array('order_id' => $entity->getOrderId(), 'status' => $params ['status']));
            }
            if($params['status'] == Order::ORDER_STATUS_CANCELED){
                $this->restoreInventory( $entity->getOrderId());
            }
            $em->flush();

        }
    }
    public function restoreInventory($order_id){

        $em = $this
            ->collection
            ->getEntityManager ();
        $order = $em->getRepository ('MotoStore\Order\Entity\Order')->findOneBy(array(
                'order_id' => $order_id)
        );
        $optRepository = $em
            ->getRepository ('MotoStore\Settings\Entity\Option');
        $trackInventoryFlag = $optRepository->getOption (Option::OPTION_STORE_SETTINGS_PRODUCT, 'track_product_inventory');
        $allowNegativeFlag  = $optRepository->getOption (Option::OPTION_STORE_SETTINGS_PRODUCT, 'allow_negative_inventory');
        $items = $order->getProducts();

        foreach ($items  as $p_product) {
            if ($p_product->getVariantId()) {
                  $variant = $em->getRepository ('MotoStore\Product\Entity\Variant')->findOneBy(array(
                                'id' => $p_product->getVariantId())
                        );
                  if ($variant){
                      $variant->trackInventory(-1*$p_product->getQuantity(), (bool)$trackInventoryFlag, (bool)$allowNegativeFlag);
                  }
            } else {
                $product = $em->getRepository ('MotoStore\Product\Entity\Product')->findOneBy(array(
                        'id' => $p_product->getProductId())
                );
                if ($product){
                    $product->trackInventory(-1*$p_product->getQuantity(), $trackInventoryFlag, $allowNegativeFlag);
                }
            }
        }

    }
    /**
     * Export Collection
     * @return array
     */

    public function export ($dates, $common = false)
    {
        $exportdir = Moto\System::getAbsolutePath($this->globalConfig['settings']['exportPath']);
        $exported =  $this->getPrefix () . '-' . date('Y_m_d_H_i_s') . '.csv';
        if ($common){
            $adapter = new OrderExportCommon ($exportdir . $exported, $this->collection, $dates);
        } else {
            $adapter = new OrderExport ($exportdir . $exported, $this->collection, $dates);
        }
        $adapter
            ->export ();
        return array (
            'exported'=>  Moto\System::getAbsoluteUrl($this->globalConfig['settings']['exportPath'] . $exported)
        );
    }
}