<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Payment\Gateway;


use MotoStore\Order\Cart\Storage;
use MotoStore\Order\Entity\Order;
use MotoStore\Settings\Entity\Currency;
use MotoStore\Settings\Entity\Option;
use Zend\Http\PhpEnvironment\Request;
use Zend\ServiceManager\ServiceLocatorInterface;

abstract class PaymentModule implements PaymentGatewayInterface
{
    const VARIABLE_TYPE_TEXT        = 'text';
    const VARIABLE_TYPE_SELECT      = 'select';
    const VARIABLE_TYPE_PASSWORD    = 'password';

    const OPTION_DISPLAY_NAME   = 'display_name';
    const OPTION_ENABLED        = 'enabled';


    /**
     * @var ServiceLocatorInterface
     */
    protected $serviceLocator;

    /**
     * @var array
     */
    protected $options = array ();

    /**
     * Payment Variables
     * @var array
     */
    protected $variables;

    /**
     * @var Storage
     */
    private $cart;

    private $errors = array ();

    /**
     * @param ServiceLocatorInterface $serviceLocator
     */
    public function __construct (ServiceLocatorInterface $serviceLocator)
    {
        $this->serviceLocator = $serviceLocator;

        $this
            ->setVariables ();
    }

    /**
     * Payment Variables Array
     * @return array
     */
    public function getVariables ()
    {
        return $this->variables;
    }

    /**
     * Add payment variable
     *
     * @param $name
     * @param array $options
     * @return $this
     */
    public function addVariable ($name, array $options)
    {
        $options ['key'] = $name;
        $this->variables [$name] = $options;

        return $this;
    }

    /**
     * Get payment variable
     */
    public function getVariable ($name, $default=null)
    {
        if (isset ($this->variables [$name]))
        {
            return $this->variables [$name];
        }
        return $default;
    }

    /**
     * Proceed payment
     *
     * @param Order $order
     * @return mixed
     */
    public abstract function checkout (Order $order);

    /**
     * Handle Success
     *
     * @return mixed
     */
    public abstract function handle (Request $request);


    /**
     * @param $key
     * @param null $default
     * @return string
     */
    public function getOption ($key, $default = null)
    {
        if (array_key_exists ($key, $this->getOptions ()))
        {
            return $this->options [$key];
        }

        return $default;
    }

    /**
     * @return string
     */
    public function getOptions ()
    {
        if (!$this->options)
        {
            $this->options = $this->getPaymentOptions ();
        }

        return $this->options;
    }

    /**
     * Load payment options
     */
    protected function getPaymentOptions ()
    {
        return $this
            ->serviceLocator
            ->get ('MotoStore\EntityManager')
            ->getRepository ('MotoStore\Settings\Entity\Option')
            ->getModuleOptions ($this->getName ());
    }

    /**
     * Return Payment Form
     *
     * @param $actionUrl
     * @param $data
     * @return string
     */
    public function form ($actionUrl, $data)
    {
        return $this
            ->serviceLocator
            ->get ('store.twig')
            ->render ('store/payment/form.twig', array (
                'url'   => $actionUrl,
                'data'  => $data,
            ));
    }

    /**
     * @param null $force_order_id
     * @return Order
     */
    protected function getCurrentOrder ($force_order_id = null)
    {
        $em = $this
            ->serviceLocator
            ->get('MotoStore\EntityManager');

        $order_id = $force_order_id ? $force_order_id : $this->getCart ()->getLastOrderId ();

        return $em
            ->getRepository ('MotoStore\Order\Entity\Order')
            ->findOneBy (array ('order_id' => $order_id));
    }

    /**
     * Is Payment Method Enabled
     *
     * @return bool
     */
    public function isEnabled ()
    {
        return (bool) $this->getOption (self::OPTION_ENABLED);
    }

    /**
     * @return Currency
     */
    public function getCurrency ()
    {
        $id = $this->getCart ()->getCurrency ();

        $em = $this->serviceLocator
            ->get ('MotoStore\EntityManager');

        if (!$id)
        {
            $id = $em
                ->getRepository('MotoStore\Settings\Entity\Option')
                ->getOption(Option::OPTION_STORE_SETTINGS_PRODUCT, 'default_currency');
        }
        return  $em
            ->getRepository ('MotoStore\Settings\Entity\Currency')
            ->find ($id);
    }

    /**
     * @return Storage
     */
    public function getCart ()
    {
        if (!$this->cart)
        {
            $this->cart = new Storage ();
        }

        return $this->cart;
    }

    public function getErrors ()
    {
        return $this->errors;
    }

    public function addError ($message)
    {
        $this->errors [] = $message;
    }

    /**
     * Debug Payment Data
     *
     * @param array $data
     * @param string $additionalMassage
     */
    public function debug ($data = array (), $additionalMassage = '')
    {
        error_log (sprintf ('%s %s:%s', $this->getName (), $additionalMassage,  var_export($data, true)));
    }
}