<?php
/***********************************************************************************************************************
 * @author: <for.samohin@gmail.com>
 **********************************************************************************************************************/

namespace MotoStore\Payment\PaypalExpress;


use MotoStore\Content\Helper\Link;
use MotoStore\Order\Entity\Order;
use MotoStore\Order\Entity\Transaction;
use MotoStore\Payment\Gateway\PaymentModule;
use Zend\Http\Client;
use Zend\Http\PhpEnvironment\Request;
use Zend\Uri\UriFactory;

class PaypalExpress extends PaymentModule
{

    /**
     * Get payment processor
     *
     * @return string
     */
    public function getName()
    {
        return 'PaypalExpress';
    }

    /**
     * Get Payment Icon
     * @return string
     */
    public function getImage ()
    {
        return 'logo-paypal.jpg';
    }

    /**
     * description
     * @return string
     */
    public function getDescription ()
    {
        return '';
    }

    /**
     * Help text
     * @return string
     */
    public function getHelpText()
    {
        return "To integrate PayPal Express Checkout into your store you need to follow a few simple steps, which are shown below:
                <ul>
                    <li>
                        <a href='https://www.paypal.com/cgi-bin/webscr?cmd=_registration-run' target=_blank>Click here</a> to register for a Paypal Express Checkout account.
                    </li>
                    <li>
                        <a href='http://www.paypal.com/' target=_blank>Click here</a> to log in to your PayPal Business account.</a>
                    </li>
                    <li>In your Paypal Business account control panel
                        <ul>
                            <li>Go to <a href='https://www.paypal.com/businessmanage/credentials/apiaccess' target=_blank>Api Access Section.</a></li>
                            <li>Scroll to  NVP / SOAP API integration (Classic) section</li>
                            <li>Click on 'Manage API credentials' in this section, then you will be able to copy your API Username, Password and Signature</li>
                        </ul>
                    </li>
                    <li>Copy and paste your API username, password, and signature to the form below.</li>
                </ul>";
    }

    /**
     * Set Payment Variables Array
     * @return array
     */
    public function setVariables ()
    {
        $this
            ->addVariable (
                self::OPTION_DISPLAY_NAME,
                array(
                    'name' => 'Display Name',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => 'PayPal Express',
                    'required' => true,
                )
            )
            ->addVariable (
                'username',
                array(
                    'name' => 'Username',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'password',
                array(
                    'name' => 'Password',
                    'type' => self::VARIABLE_TYPE_PASSWORD,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'signature',
                array(
                    'name' => 'Signature',
                    'type' => self::VARIABLE_TYPE_PASSWORD,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'transactiontype',
                array(
                    'name' => 'Transaction Type',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 'sale',
                    'required' => true,
                    'options' => array (
                        'sale' => 'Sale',
                        'authorization' => 'Authorization'
                    ),
                )
            )
            ->addVariable (
                'testmode',
                array(
                    'name' => 'Test Mode',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            )
            ->addVariable (
                self::OPTION_ENABLED,
                array(
                    'name' => 'Enabled',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            );
    }

    /**
     * Handle Success
     *
     * @return mixed
     */
    public function handle (Request $request)
    {
        $order = $this->getCurrentOrder ();
        if ($request->getQuery ('motoorderid')){
            $order = $this->getCurrentOrder ($request->getQuery ('motoorderid'));
        }
        $em = $this->serviceLocator->get ('MotoStore\EntityManager');
        if ($order)
        {
            $paymentData = $this->createCheckoutRequest ($order);

            $paymentData ['PAYERID']    = $request->getQuery ('PayerID');
            $paymentData ['TOKEN']      = $request->getQuery ('token');
            $paymentData ['METHOD']     = 'DoExpressCheckoutPayment';

            $response = $this
                ->send ($this->getTransactionUrl (), $paymentData);

            if (strtolower ($response['ACK']) == 'success')
            {
                $status = Transaction::PAYMENT_STATUS_PAID;
                $orderStatus = Order::ORDER_PAYMENT_ACCEPTED;
                $this->getCart ()->setIsSuccess (true);
                \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_PAID",$order->getOrderId());
            }
            else
            {
                //error_log ('PayPal Err: ' . json_encode($response));

                $this->retrieveErrors ($response);

                $status = Transaction::PAYMENT_STATUS_DECLINED;
                $orderStatus = Order::ORDER_STATUS_CANCELED;
            }

            $transaction = new Transaction ();
            $transaction->setOrder ($order);
            $transaction->setTransactionId (isset ($response ['TRANSACTIONID']) ? $response ['TRANSACTIONID'] : '');


            $date = \DateTime::createFromFormat('Y-m-d\TH:i:s\Z', $response ['TIMESTAMP']);


            $transaction->setDate ($date);
            $transaction->setPaymentMethod($this->getName ());
            $transaction->setStatus ($status);
            $transaction->setAdditional(json_encode ($response));

            $em->persist ($transaction);

            $order->addTransaction ($transaction);
            $order->setStatus ($orderStatus);

            $em->merge ($order);
            $em->flush ();
        }
    }


    /**
     * Proceed payment
     *
     * @param Order $order
     * @return mixed
     */
    public function checkout (Order $order)
    {
        $response = $this
            ->send ($this->getTransactionUrl (), $this->createCheckoutRequest ($order));

        if (isset ($response ['ACK']) && strtolower ($response ['ACK']) == 'success')
        {
            $url = UriFactory::factory ($this->getCheckoutUrl ());
            $url
                ->setQuery (array (
                    'token'         => $response ['TOKEN'],
                    'useraction'    => 'continue',
                    'cmd'=> '_express-checkout'
                ));

            header ('Location: ' . $url->toString ());
            exit;
        }
        else
        {
            //error_log ('PayPal Err Request: ' . json_encode($this->createCheckoutRequest ($order)));
            //error_log ('PayPal Err Response: ' . json_encode($response));
            // Order Failed
            Link::redirect ('checkout', 'success');
        }
    }

    /**
     * Send ping to paypal
     *
     * @param $url
     * @param array $data
     * @return array
     */
    protected function send ($url, $data = array ())
    {
        $result = array ();

        $client = new Client ($url);
        $client
            ->setOptions (array (
                'maxredirects'  => 0,
                'timeout'       => 60,
                'sslverifypeer' => false,
            ))
            ->setMethod (Request::METHOD_POST)
            ->setParameterPost ($data);

        $response = $client->send ();

        if ($response->isSuccess ())
        {
            parse_str ($response->getBody (), $result);
        }
        return $result;
    }

    /**
     * Create Payment Data by Order
     *
     * @param Order $order
     * @return array
     */
    protected function createCheckoutRequest (Order $order)
    {
        $currency = $this->getCurrency ();

        if ($order->getState ())
        {
            $paymentData ['SHIPTOSTATE'] = $order->getState ()->getCode ();
        }

        $names = array ();

        foreach ($order->getProducts () as $product)
        {
            $names [] = $product->getName ();
        }

        $paymentData = array (
            'SOLUTIONTYPE'  => 'Sole',
            'LANDINGPAGE'   => 'Billing',
            'METHOD'        => 'SetExpressCheckout',
            'USER'			=> $this->getOption ('username'),
            'PWD'			=> $this->getOption ('password'),
            'SIGNATURE'		=> $this->getOption ('signature'),
            'VERSION'		=> '53.0',
            'BUTTONSOURCE'		=> 'MotoCMS_SP',
            'PAYMENTACTION' => $this->getOption ('transactiontype'),
            'AMT'           => number_format ($order->getTotalPrice () * $currency->getExchangeRate (), 2, '.', ''),
            'CURRENCYCODE'  => $currency->getCode (),
            'RETURNURL'		=> Link::storeLink ('checkout', 'success', array ('provider' => $this->getName (), 'motoorderid' => $order->getOrderId ()), false),
            'CANCELURL'		=> Link::storeLink ('cart', null, array(), false),
            'ADDRESSOVERRIDE'=> 1,
            'NOTIFYURL'		=> Link::storeLink ('payments', null, array ('provider' => $this->getName (), 'motoorderid' => $order->getOrderId ()), false),
            'L_NAME0'		=> 'Order #' . $order->getOrderId () . ' [' .  join (', ', $names) . ']',
            'L_AMT0'		=> number_format ($order->getTotalPrice () * $currency->getExchangeRate (), 2, '.', ''),
            'L_QTY0'		=> 1,
        );

        return $paymentData;
    }

    /**
     * Checkout URL
     * @return string
     */
    protected function getCheckoutUrl ()
    {
        return $this->getOption ('testmode')
            ? 'https://www.sandbox.paypal.com/cgi-bin/webscr'
            : 'https://www.paypal.com/cgi-bin/webscr';
    }

    /**
     * Transaction Url
     * @return string
     */
    protected function getTransactionUrl ()
    {
        return $this->getOption ('testmode')
            ? 'https://api-3t.sandbox.paypal.com/nvp'
            : 'https://api-3t.paypal.com/nvp';
    }


    /**
     * Retrieve errors
     * @param array $response
     */
    protected function retrieveErrors (array $response)
    {
        foreach ($response as $k => $message)
        {
            if (strpos ($k, 'L_LONGMESSAGE') !== false)
            {
                $this->addError ($message);
            }
        }
    }
}