<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com>
 **********************************************************************************************************************/

namespace MotoStore\Payment\QuickPay;


use MotoStore\Content\Helper\Link;
use MotoStore\Order\Entity\Order;
use MotoStore\Order\Entity\Product;
use MotoStore\Order\Entity\Transaction;
use MotoStore\Payment\Gateway\PaymentModule;
use Zend\Http\Client;
use Zend\Http\PhpEnvironment\Request;
use Zend\Uri\UriFactory;

class QuickPay extends PaymentModule
{

    /**
     * Get payment processor
     *
     * @return string
     */
    public function getName()
    {
        return 'QuickPay';
    }

    /**
     * Get Payment Icon
     * @return string
     */
    public function getImage ()
    {
        return 'logo-quickpay.jpg';
    }

    /**
     * description
     * @return string
     */
    public function getDescription ()
    {
        return '';
    }

    /**
     * Help text
     * @return string
     */
    public function getHelpText()
    {
        return "To integrate QuickPay into your store you need to follow a few simple steps, which are shown below:<ul><li><a href='https://manage.quickpay.net/' target=_blank>Click here</a> to register for a  QuickPay account.</li><li>In your QuickPay account control panel<ul><li>Go to Settings -> Integration.</li><li>Then you should see your APIinformation:  Mercant ID, Payment Window AGREEMENT ID, Payment Window API KEY, and Merchant Private key</li><li>Copy and paste mentioned information to the form below.</li></ul>";
    }


    /**
     * Set Payment Variables Array
     * @return array
     */
    public function setVariables ()
    {
        $this
            ->addVariable (
                self::OPTION_DISPLAY_NAME,
                array(
                    'name' => 'Display Name',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => 'QuickPay',
                    'required' => true,
                )
            )
            ->addVariable (
                'merchant_id',
                array(
                    'name' => 'Merchant ID',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            ) ->addVariable (
                'agreement_id',
                array(
                    'name' => 'Agreement ID',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'api_key',
                array(
                    'name' => 'API key for Payment Window',
                    'type' => self::VARIABLE_TYPE_PASSWORD,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'api_key_merchant',
                array(
                    'name' => 'Merchant Private key',
                    'type' => self::VARIABLE_TYPE_PASSWORD,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                self::OPTION_ENABLED,
                array(
                    'name' => 'Enabled',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            );
    }

    /**
     * Handle Success
     *
     * @return mixed
     */
    public function handle (Request $request)
    {
        $order = $this->getCurrentOrder ();
        if (isset($_GET['motoorderid'])){
            $order = $this->getCurrentOrder ($_GET['motoorderid']);
        }
        $em = $this->serviceLocator->get ('MotoStore\EntityManager');
        if (isset($request) && isset($order) && $request->getQuery ('motoorderid') && isset($_SERVER["HTTP_QUICKPAY_CHECKSUM_SHA256"]))
        {
            $request_body = file_get_contents("php://input");
            $checksum     = $this->signResponce($request_body, $this->getOption ('api_key_merchant'));

            if ( isset($_SERVER["HTTP_QUICKPAY_CHECKSUM_SHA256"]) && $checksum == $_SERVER["HTTP_QUICKPAY_CHECKSUM_SHA256"]) {
                $qprequest = (array)json_decode($request_body);
                $operations = (array)$qprequest['operations'];
                foreach ($operations as $operation){
                    $operation = (array)$operation;
                    $status  = $operation['qp_status_code'];
                    $invoice_id = $qprequest['order_id'];
                    $transaction = new Transaction ();
                    $transaction->setOrder ($order);
                    $transaction->setTransactionId ($invoice_id);
                    $date = new \DateTime ();
                    $transaction->setDate ($date);
                    $transaction->setPaymentMethod($this->getName ());
                    switch ($status)
                    {
                        // Accepted
                        case 20000:
                            $current_status = $order->getStatus();
                            if ($current_status !== Order::ORDER_PAYMENT_ACCEPTED){
                                \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_PAID",$order->getOrderId());
                            }
                            $transactionStatus  = Transaction::PAYMENT_STATUS_PAID;
                            $orderStatus        = Order::ORDER_PAYMENT_ACCEPTED;

                            break;
                        // Declined
                        case 40000:
                            $transactionStatus  = Transaction::PAYMENT_STATUS_DECLINED;
                            $orderStatus        = Order::ORDER_STATUS_CANCELED;
                            break;
                        // Pending
                        case 30100:
                        case 40001:
                        case 50000:
                        case 50300:
                            $transactionStatus  = Transaction::PAYMENT_STATUS_PENDING;
                            $orderStatus        = Order::ORDER_STATUS_AWAITING_FULFILLMENT;
                            break;
                        default:
                            $transactionStatus  = Transaction::PAYMENT_STATUS_PENDING;
                            $orderStatus        = Order::ORDER_STATUS_AWAITING_FULFILLMENT;
                    }
                    $transaction->setStatus ($transactionStatus);
                    $order->setStatus ($orderStatus);
                    $transaction->setAdditional(json_encode ($qprequest));
                    $em->persist ($transaction);
                    $order->addTransaction ($transaction);
                }



                $em->merge ($order);
                $em->flush ();
                $this->getCart ()->setIsSuccess (true);
            }
        }
        $this->getCart ()->setIsSuccess (true);
    }

    /**
     * Proceed payment
     *
     * @param Order $order
     * @return mixed
     */
    public function checkout (Order $order)
    {
        $names = array_map (function (Product $p) {
            return $p->getName();
        }, $order->getProducts ()->toArray ());

        $params = array(
            "version"      => "v10",
            "merchant_id"  => $this->getOption ('merchant_id'),
            "agreement_id" => $this->getOption ('agreement_id'),
            "autocapture"     => 1,
            "order_id"     => $order->getOrderId(),
            "amount"       => rtrim(rtrim($order->getTotalPrice (), ''), '.')*100,
            "currency"     => strtoupper ($this->getCurrency()->getCode ()),
            "continueurl" => Link::storeLink ('checkout', 'success', array ('provider' => $this->getName (), 'motoorderid' => $order->getOrderId ()), false),
            "cancelurl"   => Link::storeLink ('cart' , null, array(), false),
            "callbackurl" => Link::storeLink ('checkout', 'success', array ('provider' => $this->getName (), 'motoorderid' => $order->getOrderId ()), false),
            "description" => join(', ', $names),
            "invoice_address[name]" =>  $order->getFirstName() . ' ' . $order->getLastName(),
            "invoice_address[att]" => $order->getAddress1(),
            "invoice_address[street]" => $order->getAddress2(),
            "invoice_address[phone_number]" => $order->getContactPhone(),
            "invoice_address[zip_code]" => $order->getPostCode(),
            "invoice_address[city]" => $order->getCity(),
            "invoice_address[country_code]" => $order->getCountry()->getIso3(),
            "customer_email" => $order->getEmail(),
        );
        $data['order_id'] = $params['order_id'];
        $data['version'] = $params['version'];
        $data['merchant_id'] = $params['merchant_id'];
        $data['agreement_id'] = $params['agreement_id'];
        $data['autocapture'] = $params['autocapture'];
        $data['amount'] = $params['amount'];
        $data['currency'] = $params['currency'];
        $data['continueurl'] = $params['continueurl'];
        $data['cancelurl'] = $params['cancelurl'];
        $data['callbackurl'] = $params['callbackurl'];
        $data['description'] = $params['description'];
        $data['invoice_address[name]'] = $params['invoice_address[name]'];
        $data['invoice_address[att]'] = $params['invoice_address[att]'];
        $data['invoice_address[street]'] = $params['invoice_address[street]'];
        $data['invoice_address[phone_number]'] = $params['invoice_address[phone_number]'];
        $data['invoice_address[zip_code]'] =$params['invoice_address[zip_code]'];
        $data['invoice_address[city]'] = $params['invoice_address[city]'];
        $data['invoice_address[country_code]'] = $params['invoice_address[country_code]'];
        $data['customer_email'] = $params['customer_email'];
        $data["checksum"] = $this->sign($params, $this->getOption ('api_key'));
        echo $this->form('https://payment.quickpay.net', $data);
    }

    function sign($params, $api_key) {
        $flattened_params = $this->flatten_params($params);
        ksort($flattened_params);
        $base = implode(" ", $flattened_params);

        return hash_hmac("sha256", $base, $api_key);
    }

    function flatten_params($obj, $result = array(), $path = array()) {
        if (is_array($obj)) {
            foreach ($obj as $k => $v) {
                $result = array_merge($result, $this->flatten_params($v, $result, array_merge($path, array($k))));
            }
        } else {
            $result[implode("", array_map(function($p) { return "[{$p}]"; }, $path))] = $obj;
        }

        return $result;
    }

    function signResponce($base, $private_key) {
        return hash_hmac("sha256", $base, $private_key);
    }


}