<?php

/***********************************************************************************************************************
 * @author: <for.samohin@gmail.com>
 **********************************************************************************************************************/
namespace MotoStore\Payment\Sogecommerce;

use MotoStore\Content\Helper\Link;
use MotoStore\Order\Entity\Order;
use MotoStore\Order\Entity\Transaction;
use MotoStore\Payment\Gateway\PaymentModule;
use Zend\Http\PhpEnvironment\Request;

class Sogecommerce extends PaymentModule
{

    private $_paymentUrl = 'https://sogecommerce.societegenerale.eu/vads-payment/';

    public function getName()
    {
        return 'Sogecommerce';
    }

    public function getImage()
    {
        return 'sogecommerce.png';
    }

    public function getDescription()
    {
        return '';
    }

    public function getHelpText()
    {
        return 'ShopID and keys are provided by Sogecommerce (available in Sogecommerce Back Office).<br/>
        Please, set Notification URL in Back Office in following format : \'http(s)://YOUR.DOMAIN/store/checkout/success?provider=Sogecommerce\' <br/>
        Please, set Notification method to POST';
    }

    public function setVariables()
    {
        $this
            ->addVariable (
                self::OPTION_DISPLAY_NAME,
                array(
                    'name' => 'Display Name',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => 'Sogecommerce',
                    'required' => true,
                )
            )->addVariable (
                'shop_id',
                array(
                    'name' => 'ShopID',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '12345678',
                    'required' => true,
                )
            )->addVariable (
                'key_test',
                array(
                    'name' => 'Certificate in test mode',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => sprintf('Certificate provided by %s for test mode (available in %s Back Office).', 'Sogecommerce', 'Sogecommerce'),
                    'default' => '1111111111111111',
                    'required' => false,
                )
            )->addVariable (
                'key_prod',
                array(
                    'name' => 'Certificate in production mode',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => sprintf('Certificate provided by %s (available in %s Back Office after enabling production mode).', 'Sogecommerce', 'Sogecommerce'),
                    'default' => '2222222222222222',
                    'required' => true,
                )
            )->addVariable (
                'ctx_mode',
                array(
                    'name' => 'Mode',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => 'The context mode of this module',
                    'default' => 'TEST',
                    'required' => true,
                    'options' => array (
                        'TEST' => 'TEST',
                        'PRODUCTION' => 'PRODUCTION'
                    ),
                )
            )->addVariable (
                self::OPTION_ENABLED,
                array(
                    'name' => 'Enabled',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            );
            
    }

    /**
     * @param Order $order
     * @return mixed|void
     */
    public function checkout(Order $order)
    {
        $timestamp = time();
        $data['vads_action_mode'] = "INTERACTIVE";
        $data['vads_amount'] = rtrim(rtrim($order->getTotalPrice (), '0'), '.');
        $data['vads_ctx_mode'] = $this->getOption ('ctx_mode');
        $data['vads_currency'] = $this->adaptCurrency(strtoupper ($this->getCurrency()->getCode ()));
        $data['vads_cust_country'] = $order->getCountry()->getIso2();
        $data['vads_cust_email'] = $order->getEmail();
        $data['vads_page_action'] = "PAYMENT";
        $data['vads_payment_config'] = "SINGLE";
        $data['vads_site_id'] = $this->getOption ('shop_id');
        $data['vads_trans_date'] = gmdate('YmdHis', $timestamp);
        $data['vads_trans_id'] = $order->getOrderId();
        $data['vads_version'] = "V2" ;
        $data['payer'] = $order->getFirstName() . ' ' . $order->getLastName();
        $data['signature'] = $this->getSignature($data) ;
        echo $this->form($this->_paymentUrl, $data);
    }


    public function handle(Request $request)
    {
        $payment_id = $request->getPost('vads_trans_id' );
        $notification = $request->getPost();
        $calculated_signature =  $this->getSignature($notification);
        if (!empty($payment_id) && $calculated_signature == $notification['signature'] )
        {
            $em = $this->serviceLocator->get ('MotoStore\EntityManager');
            $order = $this->getCurrentOrder ($payment_id);

            $transaction = new Transaction();
            $transaction->setOrder ($order);
            $transaction->setTransactionId ($payment_id);

            $date = new \DateTime ();

            $transaction->setDate ($date);
            $transaction->setPaymentMethod($this->getName ());


            $transaction->setStatus (Transaction::PAYMENT_STATUS_DECLINED);

            if ($notification['vads_result'] == '00')
            {
                $transaction->setStatus (Transaction::PAYMENT_STATUS_PAID);
                $order->setStatus (Order::ORDER_PAYMENT_ACCEPTED);
                $this->getCart ()
                    ->setIsSuccess (true);
                \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_PAID",$order->getOrderId());
            }

            $transaction->setAdditional(json_encode ($notification));

            $em->persist ($transaction);

            $order->addTransaction ($transaction);

            $em->merge ($order);
            $em->flush ();

        }
    }

    public function adaptCurrency($inCode)
    {
        $currenciesConversion = array(
            'ARS'=>'032',
            'AUD'=>'036',
            'CAD'=>'124',
            'HRK'=>'191',
            'CZK'=>'203',
            'DKK'=>'208',
            'EKK'=>'233',
            'HKD'=>'344',
            'HUF'=>'348',
            'LVL'=>'428',
            'LTL'=>'440',
            'MYR'=>'458',
            'MXN'=>'484',
            'NZD'=>'554',
            'NOK'=>'578',
            'PHP'=>'608',
            'RUB'=>'643',
            'SGD'=>'702',
            'ZAR'=>'710',
            'SEK'=>'752',
            'CHF'=>'756',
            'THB'=>'764',
            'GBP'=>'826',
            'USD'=>'840',
            'RON'=>'946',
            'TRY'=>'949',
            'BGN'=>'975',
            'EUR'=>'978',
            'PLN'=>'985',
            'BRL'=>'986'
        );
        if (in_array($inCode,$currenciesConversion)){
            return $currenciesConversion[$inCode];
        } else {
            return$currenciesConversion['EUR'];
        }
    }

    public function getSignature($params){
        $content_signature = "" ;
        $key = $this->getOption ('key_prod');
        if ($this->getOption ('ctx_mode') == 'TEST'){
            $key = $this->getOption ('key_test');
        }
        ksort($params);
        foreach ($params as $n =>$value)
        {

            if (substr($n,0,5)=='vads_') {
                $content_signature .= $value."+";
            }
        }
        $content_signature .= $key;
        $signature = sha1($content_signature);
        return $signature ;
    }


}