<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Payment\TwoCheckout;


use MotoStore\Order\Entity\Order;
use MotoStore\Order\Entity\Product;
use MotoStore\Order\Entity\Transaction;
use MotoStore\Payment\Gateway\PaymentModule;
use Zend\Http\PhpEnvironment\Request;

class TwoCheckout extends PaymentModule
{
    /**
     * Get payment processor
     *
     * @return string
     */
    public function getName ()
    {
        return 'TwoCheckout';
    }

    /**
     * Get Payment Icon
     * @return string
     */
    public function getImage ()
    {
        return '2checkout.jpg';
    }

    /**
     * description
     * @return string
     */
    public function getDescription ()
    {
        return '';
    }

    /**
     * Help text
     * @return string
     */
    public function getHelpText()
    {
        return "To integrate 2Checkout into your store you need to follow a few simple steps, which are shown below:<ul><li><a href=http://www.2checkout.com target=_blank>Register for a 2Checkout account here</a></li><li>Login to your account and copy the number in the heading of the page (such as <em>Vendor <strong>1234567</strong></em>) into the 'Seller ID' field below.</li><li>Click the 'Account' link in the top navigation menu and the 'Site Management' sub menu item</li><li>Under the 'Return Method' settings section, select 'Direct Return (2CO's URL)' for the 'After completing an order, buyers should be' setting</li><li>Type a word of your choice into the 'Secret Word' field and type that same word into the 'Secret Word' text box below</li><li>Paste this into the 'Approved URL' field: <textarea class='form-control' rows='3' readonly>http://mydomain.com/?url=store/checkout/success&provider=TwoCheckout</textarea></li><li>Leave all other options as they are and click the 'Save Changes' link at the bottom of the 2Checkout page</li><li>Fill in the other details below</li></ul>";
    }

    /**
     * Set Payment Variables Array
     * @return array
     */
    public function setVariables()
    {
        $this
            ->addVariable (
                self::OPTION_DISPLAY_NAME,
                array(
                    'name' => 'Display Name',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '2Checkout',
                    'required' => true,
                )
            )
            ->addVariable (
                'sellerid',
                array(
                    'name' => 'Seller ID',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'secretword',
                array(
                    'name' => 'Secret Word',
                    'type' => self::VARIABLE_TYPE_PASSWORD,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'testmode',
                array(
                    'name' => 'Test Mode',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            )
            ->addVariable (
                self::OPTION_ENABLED,
                array(
                    'name' => 'Enabled',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            );
    }

    /**
     * Handle Success
     *
     * @return mixed
     */
    public function handle (Request $request)
    {
        $order = $this->getCurrentOrder ();
        $em = $this->serviceLocator->get ('MotoStore\EntityManager');

        if ($order)
        {
            $sid            = $request->getQuery ('sid');
            $key            = $request->getQuery ('key');
            $order_number   = $request->getQuery ('order_number');
            $invoice_id     = $request->getQuery ('invoice_id');
            $total          = $request->getQuery ('total');
            $secret         = $this->getOption ('secretword', '');

            if ($this->getOption ('testmode'))
            {
                $order_number = 1;
            }

            $hash = strtoupper (md5 ($secret . $sid . $order_number . $total));

            if ($hash === $key)
            {
                $transaction = new Transaction ();
                $transaction->setOrder ($order);
                $transaction->setTransactionId ($invoice_id);


                $date = new \DateTime ();


                $transaction->setDate ($date);
                $transaction->setPaymentMethod($this->getName ());
                if ($request->getQuery ('credit_card_processed') == "Y")
                {
                    $transaction->setStatus (Transaction::PAYMENT_STATUS_PAID);
                    $order->setStatus (Order::ORDER_PAYMENT_ACCEPTED);
                    \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_PAID",$order->getOrderId());
                }
                else
                {
                    $transaction->setStatus (Transaction::PAYMENT_STATUS_DECLINED);
                }

                $transaction->setAdditional(json_encode ($request->getQuery ()));

                $em->persist ($transaction);

                $order->addTransaction ($transaction);

                $em->merge ($order);
                $em->flush ();

                $this->getCart ()->setIsSuccess (true);

            }
        }
    }




    /**
     * Proceed payment
     *
     * @param Order $order
     * @return mixed
     */
    public function checkout(Order $order)
    {
        $currency = $this->getCurrency ();

        $paymentData = array (
            'id_type'               => 1,
            'provider'              => 'TwoCheckout',
            'sid'                   => $this->getOption ('sellerid'),
            'order_id'              => $order->getOrderId (),
            'cart_order_id'         => $order->getOrderId (),
            'fixed'                 => 'Y',
            'total'                 => number_format ($order->getTotalPrice (), 2, '.', ''),
            'sh_cost'               => number_format ($order->getShippingPrice (), 2, '.', ''),


            // Billing Details
            'card_holder_name'		=> $order->getFirstName () . ' ' . $order->getLastName (),
            'street_address'		=> $order->getAddress1 (),
            'city'					=> $order->getCity (),
            'state'					=> $order->getState () ? $order->getState ()->getName () : '',
            'zip'					=> $order->getPostCode (),
            'country'				=> $order->getCountry () ? $order->getCountry ()->getName () : '',
            'phone'					=> $order->getContactPhone (),
            'email'					=> $order->getEmail (),
            'currency_code'         => $currency->getCode (),

            // Shipping Details

            'ship_name'		=> $order->getFirstName () . ' ' . $order->getLastName (),
            'ship_street_address'		=> $order->getAddress1 (),
            'ship_city'					=> $order->getCity (),
            'ship_state'				=> $order->getState () ? $order->getState ()->getName () : '',
            'ship_zip'					=> $order->getPostCode (),
            'ship_country'				=> $order->getCountry () ? $order->getCountry ()->getName () : '',
        );

        $target = 'https://www.2checkout.com/2co/buyer/purchase';

        if ($this->getOption ('testmode'))
        {
            $paymentData ['demo'] = 'Y';
            $paymentData ['cart_order_id'] = 1;
            $target = 'https://sandbox.2checkout.com/checkout/purchase';
        }

        /**
         * @var Product $product $product
         */
        foreach ($order->getProducts () as $key =>  $product)
        {
            $index = ++$key;

            $paymentData ['c_description_' . $index]	= substr ($product->getName (), 0, 254);
            $paymentData ['c_prod_' . $index]			= $product->getProductId () . ',' . $product->getQuantity ();
            $paymentData ['c_name_' . $index]			= substr ($product->getName (), 0, 128);
            $paymentData ['c_price_' . $index]		    = number_format ($product->getTotalPrice () * $currency->getExchangeRate (), 2, '.', '');
            $paymentData ['c_tangible_' . $index]	    = 'N'; // digital
        }

        echo $this
            ->form ($target, $paymentData);
    }
}