<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com>
 **********************************************************************************************************************/

namespace MotoStore\Product\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use MotoStore\Content\Helper\Price;
use MotoStore\Platform\Entity\Media;
use MotoStore\Platform\Locale\LocaleAwareInterface;
use MotoStore\Platform\Route\RoutableInterface;
use MotoStore\Product\Entity\Product\Locale;
use MotoStore\Product\Entity\Property\Relation as PropertyRelation;
use MotoStore\Settings\Entity\Currency;

/**
 * @ORM\Entity()
 * @ORM\Table(name="store_product",options={"engine":"InnoDB"})
 * @ORM\Entity(repositoryClass="MotoStore\Product\Repository\ProductRepository")
 */
class Product implements LocaleAwareInterface, RoutableInterface
{
    const PRODUCT_VISIBLE = 1;

    const PRODUCT_NOT_VISIBLE = 0;

    /**
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     * @ORM\Column(type="integer")
     */
    protected $id;

    /** @ORM\Column(type="string", nullable=true) */
    protected $sku;

    /** @ORM\Column(type="string", nullable=true) */
    protected $badge;

    /**
     * @ORM\Column(type="string")
     */
    protected $uri;

    /** @ORM\Column(type="datetime", nullable=true) */
    protected $created_date;

    /** @ORM\Column(type="string", nullable=true) */
    protected $upc;

    /** @ORM\Column(type="integer", nullable=true) */
    protected $tax_class_id;

    /** @ORM\Column(type="integer", nullable=true) */
    protected $primary_category_id;

    /** @ORM\Column(type="integer", nullable=true) */
    protected $brand_id;

    /** @ORM\Column(type="decimal", precision=10, scale=2) */
    protected $wholesale_price = 0;

    /**
     * @ORM\Column(type="decimal", precision=10, scale=2)
     */
    protected $price = 0;

    /**
     * @ORM\Column(type="decimal", precision=10, scale=2)
     */
    protected $special_price = 0;

    /** @ORM\Column(type="boolean") */
    protected $has_special_price = false;

    /** @ORM\Column(type="datetime", nullable=true) */
    protected $special_date_start;

    /** @ORM\Column(type="datetime", nullable=true) */
    protected $special_date_end;

    /**
     * @ORM\Column(type="decimal", precision=10, scale=2)
     */
    protected $weight = 0;

    /**
     * @ORM\Column(type="decimal", precision=10, scale=2)
     */
    protected $length = 0;

    /**
     * @ORM\Column(type="decimal", precision=10, scale=2)
     */
    protected $width = 0;

    /**
 * @ORM\Column(type="decimal", precision=10, scale=2)
 */
    protected $depth = 0;
    /**
     * @ORM\Column(type="decimal", precision=10, scale=2)
     */
    protected $height = 0;

    /**
     * @ORM\Column(type="integer")
     */
    protected $quantity = 0;

    /** @ORM\Column(type="boolean") */
    protected $visibility = false;

    /** @ORM\Column(type="integer") */
    protected $btob = 0;

    /** @ORM\Column(type="decimal", precision=10, scale=2) */
    protected $rating = 0;

    /**
     * @ORM\OneToMany(
     *  targetEntity="MotoStore\Product\Entity\Product\Locale",
     *  mappedBy="product",
     *  indexBy = "language_code",
     *  cascade={"persist", "remove", "merge"},
     *  orphanRemoval=true
     * )
     */
    protected $locales;

    /**
     * @ORM\ManyToOne(targetEntity="MotoStore\Platform\Entity\Media")
     * @ORM\JoinColumn(name="image_id", referencedColumnName="id", onDelete="SET NULL")
     *
     */
    protected $image;

    /**
     * @ORM\ManyToOne(targetEntity="MotoStore\Platform\Entity\Media")
     * @ORM\JoinColumn(name="file_id", referencedColumnName="id", onDelete="SET NULL")
     *
     */
    protected $file;

    /**
     * @ORM\ManyToMany(
     *  targetEntity="MotoStore\Product\Entity\Category",
     *  indexBy="id", cascade={"persist"}
     * )
     * @ORM\JoinTable(name="store_category_relation",
     *      joinColumns={@ORM\JoinColumn(name="product_id", referencedColumnName="id",onDelete="CASCADE")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="category_id", referencedColumnName="id", onDelete="CASCADE")}
     * )
     */
    protected $categories;

    /**
     * @ORM\ManyToOne(targetEntity="Brand", inversedBy="products")
     * @ORM\JoinColumn(name="brands_id", referencedColumnName="id", onDelete="SET NULL")
     *
     */
    protected $brand;

    /**
     *
     * @ORM\OneToMany(targetEntity="MotoStore\Product\Entity\Related", mappedBy="main_product", cascade={"persist", "remove"})
     * @var \Doctrine\Common\Collections\ArrayCollection
     */
    protected $related;

    /**
     * @ORM\OneToMany(
     *  targetEntity="MotoStore\Product\Entity\Property\Relation", mappedBy="product", cascade={"persist", "remove"},
     *  orphanRemoval=true
     * )
     * @ORM\OrderBy({"position" = "ASC"})
     */
    protected $properties;

    /**
     * @ORM\OneToMany(targetEntity="MotoStore\Product\Entity\Category\Relation", mappedBy="product")
     */
    protected $relations;


    /**
     * @ORM\OneToMany(
     *  targetEntity="MotoStore\Product\Entity\MediaRelation", mappedBy="product", cascade={"persist", "remove"},
     *  orphanRemoval=true
     * )
     * @ORM\OrderBy({"position" = "ASC"})
     */

    protected $images;

    /**
     * @ORM\OneToMany(
     *  targetEntity="MotoStore\Product\Entity\Variant", mappedBy="product", cascade={"persist", "remove"},
     *  orphanRemoval=true
     * )
     */


    protected $variants = array ();

    /**
     * @ORM\OneToMany(
     *  targetEntity="MotoStore\Product\Entity\Option", mappedBy="product", cascade={"persist", "remove"},
     *  orphanRemoval=true
     * )
     * @ORM\OrderBy({"position" = "ASC"})
     */
    protected $options = array ();


    /**
     * @ORM\OneToMany(
     *  targetEntity="MotoStore\Product\Entity\CustomOption", mappedBy="product", cascade={"persist", "remove"},
     *  orphanRemoval=true
     * )
     * @ORM\OrderBy({"position" = "ASC"})
     */
    protected $custom_options = array ();

    /**
     * @ORM\OneToMany(
     *  targetEntity="MotoStore\Product\Entity\TierPrice", mappedBy="product", cascade={"persist", "remove"},
     *  orphanRemoval=true
     * )
     */
    protected $tier_prices;

    /**
     * @ORM\Column(type="integer")
     */
    protected $shipping_type = 0;

    /**
     * @ORM\Column(type="decimal", precision=10, scale=2)
     */
    protected $shipping_fixed_price = 0;

    /**
     * @ORM\Column(type="string", nullable=true)
     */
    protected $video_uri;

    /**
     * @ORM\Column(type="string", nullable=true)
     */
    protected $video_uri_user;

    /** @ORM\Column(type="text", nullable=true) */
    protected $html;

    public function __construct ()
    {
        $this->locales          = new ArrayCollection ();
        $this->categories       = new ArrayCollection ();
        $this->properties       = new ArrayCollection ();
        $this->images           = new ArrayCollection ();
        $this->variants         = new ArrayCollection ();
        $this->options          = new ArrayCollection ();
        $this->custom_options   = new ArrayCollection ();
        $this->tier_prices      = new ArrayCollection ();
    }

    public function __clone() {
        $this->uri = $this->uri . '-copy-' . rand(1,99999);
        $this->visibility = false;

        /*Clone attached images records*/
        $images = $this->getImages();
        $this->images = new ArrayCollection();
        if(count($images) > 0){
            foreach ($images as $image) {
                $this->images->add($image);
            }
        }
        /*Clone locales*/
        $locales = $this->getLocales();
        $this->locales = new ArrayCollection();
        if(count($locales) > 0){
            foreach ($locales as $locale) {
                $cloneLocale = clone $locale;
                $this->locales->add($cloneLocale);
                $cloneLocale->setProduct($this);
            }
        }

        /*Clone categories relations*/
        $categories = $this->getCategories();
        $this->categories = new ArrayCollection();
        if(count($categories) > 0){
            foreach ($categories as $category) {
                $this->categories->add($category);
            }
        }

        /*Clone properties relations*/
        $properties = $this->getProperties();
        $this->properties = new ArrayCollection();
        if(count($properties) > 0){
            foreach ($properties as $property) {
                $cloneProperty = clone $property;
                $this->properties->add($cloneProperty);
                $cloneProperty->setProduct($this);
            }
        }

        /*Clone custom options*/
        $custom_options = $this->getCustomOptions();
        $this->custom_options = new ArrayCollection();
        if(count($custom_options) > 0){
            foreach ($custom_options as $custom_option) {
                $cloneOption = clone $custom_option;
                $this->custom_options->add($cloneOption);
                $cloneOption->setProduct($this);
            }
        }
        /*Clone tier_prices*/
        $tier_prices = $this->getTierPrices();
        $this->tier_prices = new ArrayCollection();
        if(count($tier_prices) > 0){
            foreach ($tier_prices as $tier_price) {
                $clonePrice = clone $tier_price;
                $this->tier_prices->add($clonePrice);
                $clonePrice->setProduct($this);
            }
        }

        /*Clone related products*/
        $related = $this->getRelated();
        $this->related = new ArrayCollection();
        if(count($related) > 0){
            foreach ($related as $r_product) {
                $cloneOption = clone $r_product;
                $this->related->add($cloneOption);
                $cloneOption->setMainProduct($this);
            }
        }

        /*Clone options*/
        $options = $this->getOptions();
        $this->options = new ArrayCollection();
        if(count($options) > 0){
            foreach ($options as $option) {
                $cloneOption = clone $option;
                $this->options->add($cloneOption);
                $cloneOption->setProduct($this);
            }
        }

        /*Clone variants*/
        $variants = $this->getVariants();
        $this->variants = new ArrayCollection();
        if(count($variants) > 0){
            foreach ($variants as $variant) {
                $cloneVariant = clone $variant;
                $this->variants->add($cloneVariant);
                $cloneVariant->setProduct($this);
            }
        }

    }

       /**
        * @param int $id
        */
    public function setId ($id)
    {
        $this->id = $id;
    }


    /**
     * @return int
     */
    public function getId ()
    {
        return $this->id;
    }


    /**
     * @param string $sku
     */
    public function setSku ($sku)
    {
        $this->sku = $sku;
    }


    /**
     * @return string
     */
    public function getSku ()
    {
        return $this->sku;
    }

    /**
     * @param int $tax_class_id
     */
    public function setTaxClassId ($tax_class_id)
    {
        $this->tax_class_id = $tax_class_id;
    }


    /**
     * @return int
     */
    public function getTaxClassId ()
    {
        return $this->tax_class_id;
    }

    /**
     * @param int $primary_category_id
     */
    public function setPrimaryCategoryId ($primary_category_id)
    {
        $this->primary_category_id = $primary_category_id;
    }


    /**
     * @return int
     */
    public function getPrimaryCategoryId ()
    {
        return $this->primary_category_id;
    }


    /**
     * @param float $price
     */
    public function setPrice ($price)
    {
        $this->price = $price;
    }

    /**
     * @return float
     */
    public function getPrice ()
    {
        return $this->price;
    }


    /**
     * @param string $upc
     */
    public function setUpc ($upc)
    {
        $this->upc = $upc;
    }


    /**
     * @return string
     */
    public function getUpc ()
    {
        return $this->upc;
    }


    /**
     * @param float $depth
     */
    public function setDepth ($depth)
    {
        $this->depth = $depth;
    }


    /**
     * @return float
     */
    public function getDepth ()
    {
        return $this->depth;
    }
    /**
     * @param float $height
     */
    public function setHeight ($height)
    {
        $this->height = $height;
    }


    /**
     * @return float
     */
    public function getHeight ()
    {
        return $this->height;
    }

    /**
     * @param float $length
     */
    public function setLength ($length)
    {
        $this->length = $length;
    }


    /**
     * @return float
     */
    public function getLength ()
    {
        return $this->length;
    }


    /**
     * @param float $weight
     */
    public function setWeight ($weight)
    {
        $this->weight = $weight;
    }


    /**
     * @return float
     */
    public function getWeight ()
    {
        return $this->weight;
    }


    /**
     * @param float $width
     */
    public function setWidth ($width)
    {
        $this->width = $width;
    }


    /**
     * @return float
     */
    public function getWidth ()
    {
        return $this->width;
    }


    /**
     * @param float $quantity
     */
    public function setQuantity ($quantity)
    {
        $this->quantity = $quantity;
    }


    /**
     * @return float
     */
    public function getQuantity ()
    {
        return $this->quantity;
    }

    /**
     * @param float $rating
     */
    public function setRating ($rating)
    {
        $this->rating = $rating;
    }


    /**
     * @return float
     */
    public function getRating ()
    {
        return $this->rating;
    }


    /**
     * @param Locale $locale
     */
    public function addLocale (Locale $locale)
    {
        $locale
            ->setProduct ($this);
        $this
            ->locales
            ->add($locale);
    }


    /**
     * @return ArrayCollection
     */
    public function getLocales ()
    {
        return $this->locales;
    }


    /**
     * @param float $special_price
     */
    public function setSpecialPrice ($special_price)
    {
        $this->special_price = $special_price;
    }


    /**
     * @return float
     */
    public function getSpecialPrice ()
    {
        return $this->special_price;
    }


    /**
     * @param mixed $brand_id
     */
    public function setBrandsId ($brand_id)
    {
        $this->brand_id = $brand_id;
    }


    /**
     * @return mixed
     */
    public function getBrandId ()
    {
        return $this->brand_id;
    }

    /**
     * @return mixed
     */
    public function getBrandsId ()
    {
        return $this->brand_id;
    }

    public function removeBrands ()
    {
       $this->brand_id = NULL;
       $this->brand = NULL;
    }


    /**
     * @param mixed $brand
     */
    public function attachBrand (Brand $brand = null)
    {
        $this->brand = $brand;
    }


    /**
     * @return Brand
     */
    public function getBrand ()
    {
        return $this->brand
            ? $this->brand
            : new Brand ();
    }


    /**
     * @param mixed $uri
     */
    public function setUri ($uri)
    {
        $this->uri = $uri;
    }


    /**
     * @return mixed
     */
    public function getUri ()
    {
        return $this->uri;
    }


    /**
     * @param bool $visibility
     */
    public function setVisibility ($visibility)
    {
        $this->visibility = $visibility;
    }


    /**
     * @return mixed
     */
    public function getVisibility ()
    {
        return $this->visibility;
    }

    /**
     * @param bool $btob
     */
    public function setBtob ($btob)
    {
        $this->btob = $btob;
    }


    /**
     * @return mixed
     */
    public function getBtob ()
    {
        return $this->btob;
    }


    /**
     * @param bool $shipping_type
     */
    public function setShippingType ($shipping_type)
    {
        $this->shipping_type = $shipping_type;
    }


    /**
     * @return mixed
     */
    public function getShippingType ()
    {
        return $this->shipping_type;
    }

    /**
     * @param bool $shipping_fixed_price
     */
    public function setShippingFixedPrice ($shipping_fixed_price)
    {
        $this->shipping_fixed_price = $shipping_fixed_price;
    }


    /**
     * @return mixed
     */
    public function getShippingFixedPrice ()
    {
        return $this->shipping_fixed_price;
    }

    /**
     * @param \DateTime $special_date_end
     */
    public function setCreatedDate (\DateTime $created_date = null)
    {
        $this->created_date = $created_date;
    }


    /**
     * @return \DateTime
     */
    public function getCreatedDate ()
    {
        return $this->created_date
            ? $this->created_date
            : new \DateTime();
    }

    /**
     * @param \DateTime $special_date_end
     */
    public function setSpecialDateEnd (\DateTime $special_date_end = null)
    {
        $this->special_date_end = $special_date_end;
    }


    /**
     * @return \DateTime
     */
    public function getSpecialDateEnd ()
    {
        return $this->special_date_end
            ? $this->special_date_end
            : new \DateTime();
    }


    /**
     * @param \DateTime $special_date_start
     */
    public function setSpecialDateStart (\DateTime $special_date_start = null)
    {
        $this->special_date_start = $special_date_start;
    }


    /**
     * @return \DateTime
     */
    public function getSpecialDateStart ()
    {
        return $this->special_date_start
            ? $this->special_date_start
            : new \DateTime ();
    }


    /**
     * @param mixed $wholesale_price
     */
    public function setWholesalePrice ($wholesale_price)
    {
        $this->wholesale_price = $wholesale_price;
    }


    /**
     * @return mixed
     */
    public function getWholesalePrice ()
    {
        return $this->wholesale_price;
    }

    /**
     * @return mixed
     */
    public function getHasSpecialPrice()
    {
        return $this->has_special_price;
    }

    /**
     * @param mixed $has_special_price
     */
    public function setHasSpecialPrice($has_special_price)
    {
        $this->has_special_price = $has_special_price;
    }

    /**
     * If isset special and it allowed return special else default
     * @return float
     */
    public function getFinalPrice ()
    {
        $price = $this->getPrice ();

        if ($this->getHasSpecialPrice () )
        {
            $currentDate = new \DateTime ();

            if ($this->getSpecialDateStart () && $this->getSpecialDateEnd ())
            {
                if ($currentDate > $this->getSpecialDateStart () && $currentDate < $this->getSpecialDateEnd ())
                {
                    $price = $this->getSpecialPrice ();
                }
            }
        }
        return (float) $price;
    }

    /**
     * @param $quantity
     * @return float
     */
    public function getSubtotalPrice ($quantity)
    {
        return $this->getFinalPrice () * $quantity;
    }

    /**
     * @param $code
     * @return null|Locale
     */
    public function getLocaleByLanguageCode ($code)
    {
        $code = strtolower ($code);

        foreach ($this->getLocales () as $locale)
        {
            if (strtolower ($locale->getLanguageCode ()) == $code)
            {
                return $locale;
            }
        }
        return new Locale ($code);
    }


    /**
     * @return ArrayCollection
     */
    public function getProperties ()
    {
        return $this->properties;
    }


    /**
     * @param $property
     */
    public function addProperty ($property)
    {
        $this->properties->add ($property);
    }

    public function removeProperty (PropertyRelation $relation)
    {
        $this->properties->removeElement ($relation);
    }

    /**
     * @return string
     */
    public function getBadge ()
    {
        return $this->badge;
    }


    /**
     * @param mixed $badge
     */
    public function setBadge ($badge)
    {
        $this->badge = $badge;
    }

    /**
     * @return Media
     */
    public function getImage()
    {
        return $this->image;
    }

    /**
     * @param mixed $image
     */
    public function attachMainImage (Media $image = null)
    {
        $this->image = $image;
    }

    /**
     * @return Media
     */
    public function getFile()
    {
        return $this->file;
    }

    /**
     * @param mixed $file
     */
    public function attachFile (Media $file = null)
    {
        $this->file = $file;
    }
    /**
     *
     */
    public function removeFile ()
    {

            $this->file = null;

    }

    /**
     * @return ArrayCollection
     */
    public function getImages ()
    {
        return $this->images;
    }
    public function getImagesSingle ($id)
    {
        return $this
            ->images
            ->filter (
                function ( $image) use ($id) {
                    return $image->getId() == $id;
                }
            )->first ();
    }
    /**
     * @param mixed $images
     */
    public function attachImages ($images)
    {
        foreach ($images as $image)
        {
            if (!$this->images->contains ($image))
            {
                $this->images->add ($image);
            }
        }
    }


    public function clearImages ()
    {
        $this->images = null;
    }

    /**
     * @param Media $image
     */
    public function removeImage (Media $image)
    {
        if ($this->images->contains ($image))
        {
            $this->images->removeElement ($image);
        }
    }

    /**
     * @param mixed $categories
     */
    public function attachCategories ($categories)
    {
        foreach ($categories as $category)
        {
            if (!$this->categories->contains ($category))
            {
                $this->categories->add ($category);
            }
        }
    }

    /**
     * @param Category $category
     */
    public function removeCategory (Category $category = null)
    {
        if ($this->categories->contains ($category))
        {
            $this->categories->removeElement ($category);
        }
    }

    /**
     * @return ArrayCollection
     */
    public function getCategories()
    {
        return $this->categories;
    }

    /**
     * @return ArrayCollection
     */
    public function getVariants()
    {
        return $this->variants;
    }

    /**
     * @param mixed $variants
     */
    public function setVariants($variants)
    {
        $this->variants = $variants;
    }

    /**
     * @return ArrayCollection
     */
    public function getOptions()
    {
        return $this->options;
    }

    /**
     *
     * @param $option_id
     * @return Option Option
     */
    public function getOption ($option_id)
    {
        return $this
            ->options
            ->filter (
                function (Option $o) use ($option_id) {
                    return $o->getId() == $option_id;
                }
            )->first ();
    }

    /**
     *
     * @param $name
     * @return Option
     */
    public function getOptionByName ($name)
    {
        return $this
            ->getOptions ()
            ->filter (
                function (Option $option) use ($name)
                {
                    return $option->getName() == $name;
                }
            )->first ();
    }

    /**
     * @param mixed $options
     */
    public function setOptions($options)
    {
        $this->options = $options;
    }

    /**
     * @return ArrayCollection
     */
    public function getCustomOptions()
    {
        return $this->custom_options;
    }

    /**
     *
     * @param $option_id
     * @return CustomOption Option
     */
    public function getCustomOption ($option_id)
    {
        return $this
            ->custom_options
            ->filter (
                function (CustomOption $o) use ($option_id) {
                    return $o->getId() == $option_id;
                }
            )->first ();
    }

    /**
     *
     * @param $name
     * @return CustomOption
     */
    public function getCustomOptionByName ($name)
    {
        return $this
            ->getOptions ()
            ->filter (
                function (CustomOption $option) use ($name)
                {
                    return $option->getName() == $name;
                }
            )->first ();
    }

    /**
     * @param mixed $options
     */
    public function setCustomOptions($custom_options)
    {
        $this->custom_options = $custom_options;
    }
    /**
     *
     * Track Inventory
     *
     * @param int $count
     * @param bool $trackInventory
     * @param bool $allowNegative
     * @return bool
     */
    public function trackInventory ($count = 0, $trackInventory = true, $allowNegative = false)
    {
        $result = true;

        if ($trackInventory)
        {
            $inventory = $this->getQuantity () - $count;

            if ($inventory >= 0 || $allowNegative)
            {
                $this->setQuantity ($inventory);
            }
            else
            {
               $result = false;
            }
        }
        return $result;
    }

    /**
     * Get Variant By ID
     *
     * @param $variant_id
     * @return Variant
     */
    public function getVariant ($variant_id)
    {
        return $this
            ->variants
            ->filter (
                function (Variant $v) use ($variant_id) {
                    return $v->getId () == $variant_id;
                }
            )->first ();
    }

    /**
     * @return ArrayCollection
     */
    public function getRelated()
    {
        return $this->related;
    }

    public function setRelated($related)
    {
        $this->related = $related;
    }

    /**
     * @param Related $product
     */
    public function removeRelated (Related $product)
    {
        if ($this->related->contains ($product))
        {
            $this->related->removeElement ($product);
        }
    }

    /**
     * @return ArrayCollection
     */
    public function getTierPrices()
    {
        return $this->tier_prices;
    }
    
    

    /**
     * @param mixed $tierPrices
     */
    public function addTierPrice ($tierPrices)
    {
        $this->tier_prices->add ($tierPrices);
    }

    /**
     * @param mixed $tierPrices
     */
    public function addTierPrices ($tierPrices)
    {
        $this->tier_prices = $tierPrices;
    }
    
    public function getTiersSetting (Currency $currency)
    {
        return array_map(function (TierPrice $tierPrice) use ($currency) {
            
            return array (
                'q' => $tierPrice->getQuantity(), 
                'p' => Price::decorate($tierPrice->getPrice (), $currency)
            );
            
        }, $this->getTierPrices ()->toArray ());
    }

    /**
     * @param string $uri
     */
    public function setVideoUri ($uri)
    {
        $this->video_uri = $uri;
    }

    /**
     * @return string
     */
    public function getVideoUri ()
    {
        return $this->video_uri;
    }

    /**
     * @param string $uri
     */
    public function setVideoUriUser ($uri)
    {
        $this->video_uri_user = $uri;
    }

    /**
     * @return string
     */
    public function getVideoUriUser ()
    {
        return $this->video_uri_user;
    }
    /**
     * @param string $html
     */
    public function setHtml ($html)
    {
        $this->html = $html;
    }

    /**
     * @return string
     */
    public function getHtml ()
    {
        return $this->html;
    }
}