<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com>
 **********************************************************************************************************************/

namespace MotoStore\Product\Service;


use MotoStore\Collection\AbstractExport;
use MotoStore\Collection\Collection;
use MotoStore\Collection\ExportInterface;
use MotoStore\Content\Content\DI\SMStatic;
use MotoStore\Product\Entity\Product;
use MotoStore\Product\Entity\Property;

class Export extends AbstractExport implements ExportInterface
{
    /**
     * CSV delimiter
     * @var string
     */
    private $delimiter = ',';

    /**
     * CSV enclosure
     * @var string
     */
    private $enclosure = '"';

    /**
     * Collection to Export
     * @var Collection
     */
    private $collection;

    /**
     * Target resource
     * @var
     */
    private $resource;
    private $settings;

    private $categories = array ();

    /**
     * AbstractExport constructor.
     * @param $target
     * @param Collection $collection
     */
    public function __construct ($target, Collection $collection, $settings)
    {
        $this->target = $target;
        $this->collection = $collection;
        $this->settings = $settings;
    }


    /**
     * CSV columns
     * @return array
     */
    public function getHeaders ()
    {

        $headers = array(
            'id',
            'type',
            'action',
            'product_id',
            'name',
            'sku',
            'upc',
            'badge',
            'brand',
            'uri',
            'created_date',
            'image',
            'tax_class_id',
            'wholesale_price',
            'price',
            'special_price',
            'has_special_price',
            'special_date_start',
            'special_date_end',
            'quantity',
            'visibility',
            'keywords',
            'meta_description',
            'short_description',
            'description',
            'video_uri',
        );

       if ($this->settings['og']){
            $headers []= 'og_image';
            $headers []= 'og_title';
            $headers []= 'og_url';
            $headers []= 'og_description';
        }

        if ($this->settings['categories']){
            $headers []= 'categories';
        }
        if ($this->settings['shipping']){
            $headers []= 'weight';
            $headers []= 'length';
            $headers []= 'width';
            $headers []= 'depth';
            $headers []= 'shipping_type';
            $headers []= 'shipping_fixed_price';
        }

        if ($this->settings['related']){
            $headers []= 'related_product_id';
            $headers []= 'related_product_name';
        }
        if ($this->settings['custom_options']){
            $headers []= 'custom_option_id';
            $headers []= 'custom_option_product_id';
            $headers []= 'custom_option_position';
            $headers []= 'custom_option_name';
            $headers []= 'custom_option_price';
            $headers []= 'custom_option_type';

        }
        if ($this->settings['variants']){
            $headers []= 'variation_id';
            $headers []= 'variation_sku';
            $headers []= 'variation_params_view_only';
            $headers []= 'variation_price';
            $headers []= 'variation_quantity';
        }
        if ($this->settings['properties']){
            $headers []= 'property_property_id';
            $headers []= 'property_value_id';
            $headers []= 'property_value';
            $headers []= 'property_position';
            $headers []= 'property_name';
        }

        return $headers;

    }


    public  function getStingGlue(){
        return $this->settings['excel_compatible']?',':';';
    }

    public  function  getValuesSeparator(){
        return $this->settings['excel_compatible']?';':',';
    }

    /**
     * Prepare record before put to csv
     *
     * @param $record
     * @return array
     */
    public function prepareRecord ($record)
    {

        $record['type'] = 'product';
        $record['action'] = 'update';
        /** @var \DateTime $specialDateStart */
        $createdDate   = $record ['created_date'];
        /** @var \DateTime $specialDateStart */
        $specialDateStart   = $record ['special_date_start'];
        /** @var \DateTime $specialDateEnd */

        $record ['has_special_price'] = (int) $record ['has_special_price'];
        $specialDateEnd     = $record ['special_date_end'];

        $record ['special_date_start'] = $specialDateStart ? $specialDateStart->format ('Y-m-d H:i:s') : '';
        $record ['special_date_end'] = $specialDateEnd ? $specialDateEnd->format ('Y-m-d H:i:s') : '';
        $record ['created_date'] = $createdDate ? $createdDate->format ('Y-m-d H:i:s') : '';
        $locale = $record ['locales'] ['en'];

        $record =  array_merge ($record, $locale);
        $record['keywords']  = str_replace(',','%2C',  $record['keywords']);
        $record['meta_description'] = str_replace(',','%2C',  $record['meta_description']);
        $record['short_description'] = str_replace(',','%2C',  $record['short_description']);
        $record['description'] = str_replace(',','%2C',  $record['description']);
        if (!empty ($record ['image']))
        {
            $record ['image'] = $record ['image']['path'];
        }

        if (!empty ($record ['brand']))
        {
            $record ['brand'] = $record ['brand']['name'];
        }

        if (is_array ($record ['images']))
        {
            $images = array_map (
                function ($img) {
                    return $img ['path'];

                },
                $record ['images']
            );

            $record ['images'] = implode ($this->getStingGlue(), $images);
        }

        if ($this->settings['categories']){
            $record ['categories'] = implode($this->getStingGlue(), $this->getCategoriesNameList (array_keys ($record ['categories'])));
        }
        if ($this->settings['og']){
            $record ['og_title'] = str_replace(',','%2C',  $record['og_title']);
            $record ['og_description'] = str_replace(',','%2C',  $record['og_description']);
            $record ['og_url'] = str_replace(',','%2C',  $record['og_url']);
            $record ['og_image'] = str_replace(',','%2C',  $record['og_image']);

        }

        $keys_to_empty = array(
            'custom_option_id',
            'custom_option_name',
            'custom_option_price',
            'custom_option_product_id',
            'custom_option_position',
            'custom_option_type',
            'variation_id',
            'variation_sku',
            'variation_params_view_only',
            'variation_price',
            'variation_quantity',
            'property_property_id',
            'property_value_id',
            'property_value',
            'property_position',
            'property_name',
            'related_product_id',
            'related_product_name',

        );
        foreach ($keys_to_empty as $key){
            if(in_array($key,$this->getHeaders())){
                $record[$key] = '';
            }

        }
        return $record;
    }

    public function prepareRecordCustomOption ($custom_option)
    {
        error_log(json_encode($custom_option));
        $record['type'] = 'custom_option';
        $record['action'] = 'update';
        $record['product_id'] = $custom_option['product_id'];
        $record['custom_option_id'] = $custom_option['id'];
        $record['custom_option_name'] = $custom_option['name'];
        $record['custom_option_price'] = $custom_option['price'];
        $record['custom_option_product_id'] = $custom_option['product_id'];
        $record['custom_option_position'] = $custom_option['position'];
        $record['custom_option_type'] = $custom_option['type'];

        $keys_to_empty = array(
            'sku',
            'upc',
            'badge',
            'brand',
            'uri',
            'created_date',
            'name',
            'image',
            'categories',
            'related',
            'tax_class_id',
            'wholesale_price',
            'price',
            'special_price',
            'has_special_price',
            'special_date_start',
            'special_date_end',
            'weight',
            'length',
            'width',
            'depth',
            'shipping_type',
            'shipping_fixed_price',
            'og_image',
            'og_title',
            'og_url',
            'og_description',
            'quantity',
            'visibility',
            'keywords',
            'meta_description',
            'short_description',
            'description',
            'variation_id',
            'variation_sku',
            'variation_params_view_only',
            'variation_price',
            'variation_quantity',
            'property_property_id',
            'property_value_id',
            'property_value',
            'property_position',
            'property_name',
            'related_product_id',
            'related_product_name',
            'video_uri',
        );
        foreach ($keys_to_empty as $key){
            if(in_array($key,$this->getHeaders())){
                $record[$key] = '';
            }
        }

        return $record;
    }

    public function prepareRecordVariation ($variant)
    {

        /** @var Product $product_entity */
        $product_entity = SMStatic::getInstance()
            ->get ('MotoStore\EntityManager')
            ->getRepository ('MotoStore\Product\Entity\Product')
            ->find ($variant['product_id']);
        $variant_options= array();
        $product_variants = $product_entity->getVariants();
        if(count($product_variants) > 0) {
            foreach ($product_variants as $pvariant){
                if ($pvariant->getId() == $variant['id']){
                    $var_options = $pvariant->getOptions();
                    if(count($var_options) > 0){
                        foreach ($var_options as $var_option) {
                            $variant_options[] = array(
                                'option' => $var_option->getOption()->getName(),
                                'value' => $var_option->getValue()->getName()
                            );
                        }
                    }
                }
            }
        }
        $record['type'] = 'variation';
        $record['action'] = 'update';
        $record['product_id'] = $variant['product_id'];
        $record['variation_id'] = $variant['id'];
        $record['variation_sku'] = $variant['sku'];
        $record['variation_params_view_only'] = json_encode($variant_options);
        $record['variation_price'] = $variant['price'];
        $record['variation_quantity'] = $variant['quantity'];
        $keys_to_empty = array(
            'sku',
            'upc',
            'badge',
            'brand',
            'uri',
            'created_date',
            'name',
            'image',
            'categories',
            'related',
            'tax_class_id',
            'wholesale_price',
            'price',
            'special_price',
            'has_special_price',
            'special_date_start',
            'special_date_end',
            'weight',
            'length',
            'width',
            'depth',
            'shipping_type',
            'shipping_fixed_price',
            'og_image',
            'og_title',
            'og_url',
            'og_description',
            'quantity',
            'visibility',
            'keywords',
            'meta_description',
            'short_description',
            'description',
            'custom_option_id',
            'custom_option_name',
            'custom_option_price',
            'custom_option_product_id',
            'custom_option_position',
            'custom_option_type',
            'property_property_id',
            'property_value_id',
            'property_value',
            'property_position',
            'property_name',
            'related_product_id',
            'related_product_name',
            'video_uri',
        );
        foreach ($keys_to_empty as $key){
            if(in_array($key,$this->getHeaders())){
                $record[$key] = '';
            }
        }
        return $record;
    }


    public function prepareRecordProperty ($property)
    {

        /** @var Property\Value $property_value */
        $property_value = SMStatic::getInstance()
            ->get ('MotoStore\EntityManager')
            ->getRepository ('MotoStore\Product\Entity\Property\Value')
            ->find ($property['value_id']);

        /** @var Property $property_entity */
        $property_entity = SMStatic::getInstance()
            ->get ('MotoStore\EntityManager')
            ->getRepository ('MotoStore\Product\Entity\Property')
            ->find ($property['property_id']);

        $record['type'] = 'property';
        $record['action'] = 'update';
        $record['product_id'] = $property['product_id'];
        $record['property_property_id'] = $property['property_id'];
        $record['property_name'] = $property_entity->getName();
        $record['property_value_id'] = $property['value_id'];
        $record['property_value'] = $property_value->getValue();
        $record['property_position'] = $property['position'];
        $keys_to_empty = array(
            'sku',
            'upc',
            'badge',
            'brand',
            'uri',
            'created_date',
            'name',
            'image',
            'categories',
            'related',
            'tax_class_id',
            'wholesale_price',
            'price',
            'special_price',
            'has_special_price',
            'special_date_start',
            'special_date_end',
            'weight',
            'length',
            'width',
            'depth',
            'shipping_type',
            'shipping_fixed_price',
            'og_image',
            'og_title',
            'og_url',
            'og_description',
            'quantity',
            'visibility',
            'keywords',
            'meta_description',
            'short_description',
            'description',
            'custom_option_id',
            'custom_option_name',
            'custom_option_price',
            'custom_option_product_id',
            'custom_option_position',
            'custom_option_type',
            'variation_id',
            'variation_sku',
            'variation_params_view_only',
            'variation_price',
            'variation_quantity',
            'related_product_id',
            'related_product_name',
            'video_uri',
        );
        foreach ($keys_to_empty as $key){
            if(in_array($key,$this->getHeaders())){
                $record[$key] = '';
            }
        }
        return $record;
    }

    public function prepareRecordRelated ($related)
    {

        $record['type'] = 'related';
        $record['action'] = 'update';
        $record['product_id'] = $related['main_product_id'];
        $record['related_product_id'] = $related['product_id'];
        $record['related_product_name'] = $related['name'];
        $keys_to_empty = array(
            'sku',
            'upc',
            'badge',
            'brand',
            'uri',
            'created_date',
            'name',
            'image',
            'categories',
            'tax_class_id',
            'wholesale_price',
            'price',
            'special_price',
            'has_special_price',
            'special_date_start',
            'special_date_end',
            'weight',
            'length',
            'width',
            'depth',
            'shipping_type',
            'shipping_fixed_price',
            'og_image',
            'og_title',
            'og_url',
            'og_description',
            'quantity',
            'visibility',
            'keywords',
            'meta_description',
            'short_description',
            'description',
            'custom_option_id',
            'custom_option_name',
            'custom_option_price',
            'custom_option_product_id',
            'custom_option_position',
            'custom_option_type',
            'variation_id',
            'variation_sku',
            'variation_params_view_only',
            'variation_price',
            'variation_quantity',
            'property_property_id',
            'property_value_id',
            'property_value',
            'property_name',
            'property_position',
            'video_uri',
        );
        foreach ($keys_to_empty as $key){
            if(in_array($key,$this->getHeaders())){
                $record[$key] = '';
            }
        }
        return $record;
    }


    /**
     * Return category name list by id's list
     *
     * @param array $ids
     * @return array
     */
    public function getCategoriesNameList ($ids = array ())
    {
        $categories = $this->getCategoriesCollection ();

        return array_map (function ($id) use ($categories) {

            return isset ($categories [$id])
                ? $categories [$id]
                : null;
        }, $ids);
    }

    public function export ()
    {
        $this->open ();
        $records_without_ids = array();
        $records = array();
        $tmp_records = $this->collection
            ->getCollection ();
        foreach ($tmp_records as $record){
            $records_without_ids[] = $this->prepareRecord($record);
            if($this->settings['custom_options']){
                foreach ($record['custom_options'] as $custom_option){
                    $records_without_ids[] = $this->prepareRecordCustomOption($custom_option);
                }
            }
            if($this->settings['variants']){
                foreach ($record['variants'] as $variant){
                    $records_without_ids[] = $this->prepareRecordVariation($variant);
                }
            }
            if($this->settings['properties']){
                foreach ($record['properties'] as $property){
                    $records_without_ids[] = $this->prepareRecordProperty($property);
                }
            }
            if($this->settings['related']){
                foreach ($record['related'] as $related){
                    $records_without_ids[] = $this->prepareRecordRelated($related);
                }
            }



        }
        foreach ($records_without_ids as $key=>$record){
            $record['id'] = $key;
            $records[] = $record;
        }

        $this->write ($this->getHeaders ());

        foreach ($records as $record)
        {
            $this->write ($this->excludeFields ($record));
        }
        $this->close ();
    }

    /**
     * Filter record by headers
     * @param $record
     * @return array
     */
    private function excludeFields ($record)
    {
        $headers = array_flip ($this->getHeaders ());

        return array_intersect_key (array_merge ($headers, $record), $headers);
    }

    /**
     * Create Target Resource
     */
    private function open ()
    {
        $this->resource = fopen ($this->target, 'w');
    }

    /**
     * Write record to CSV
     * @param array $record
     */
    private function write (array $record)
    {
        fputcsv ($this->resource, $record, $this->getValuesSeparator(), $this->enclosure);
    }

    /**
     * Close resource
     */
    private function close ()
    {
        return fclose ($this->resource);
    }

    /**
     * Fetch categories
     *
     * @return array
     */
    private function getCategoriesCollection ()
    {
        if (!$this->categories)
        {
            /** @var Collection $collection */
            $collection = SMStatic::getInstance ()->get ('store.collection.category');

            $category_collection = $collection->getCollection(function ($category) {
                return array (
                    'id' => $category ['id'],
                    'name' => $category ['locales'] ['en'] ['name'],
                );
            });

            foreach ($category_collection as $item)
            {
                $this->categories [$item ['id']] = $item ['name'];

            }

            unset ($collection);
            unset ($category_collection);
        }

        return $this->categories;
    }
}