<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Product\Service;

use Doctrine\ORM\Query;
use MotoStore\Content\Helper\Link;
use MotoStore\Content\Helper\Price;
use MotoStore\Product\Entity\Option;
use MotoStore\Product\Entity\CustomOption;
use MotoStore\Product\Entity\OptionValue;
use MotoStore\Product\Entity\Product;
use MotoStore\Product\Entity\Variant;
use MotoStore\Product\Form\ProductType;
use MotoStore\Product\Form\VariantType;
use MotoStore\Product\Repository\ProductRepository;
use MotoStore\Service\ServiceEditableInterface;
use MotoStore\Service\ServiceReadableInterface;
use MotoStore\Service\ServiceSearchableInterface;
use MotoStore\Service\StoreServiceAbstract;
use MotoStore\Content\Content\DI\SMStatic;
use MotoStore\Platform\Entity\Media as Media;
use Moto;


class ProductServiceClass extends StoreServiceAbstract
    implements ServiceReadableInterface, ServiceEditableInterface, ServiceSearchableInterface
{
    protected $_resourcePrivilegesMap = array (
        'getCollection' => 'get',
        'getProductsList' => 'get',
        'getProductsCatalog' => 'get',
        'search' => 'get',
        'getItem' => 'get',
        'getVariants' => 'get',
        'getOptions' => 'get',
        'getCustomOptions' => 'get',
        'getCurrency' => 'get',
    );

    /**
     * Prefix of Resource
     *
     * @return mixed
     */
    protected function getPrefix ()
    {
        return 'product';
    }

    /**
     * Collection Name
     *
     * @return string
     */
    protected function getCollectionName()
    {
        return 'store.collection.product';
    }

    /**
     * Save Resource
     *
     * @return mixed
     */
    public function save ()
    {
        return $this
            ->handleUpdateRequest (new ProductType (), new Product ());
    }

    /**
     * Remove Resource
     *
     * @param $id
     * @return mixed
     */
    public function delete ($id)
    {
        return $this
            ->handleDeleteRequest ($id);
    }

    /**
     * Retrieve Collection By Params
     *
     * @return mixed
     */
    public function getCollection ()
    {
        return $this
            ->handleCollectionRequest ();
    }

    public function getProductsList ()
    {
        $query = $this
            ->collection
            ->getEntityManager ()
            ->createQueryBuilder ();

        $products = $query
            ->select(array ('p','l'))
            ->from ('MotoStore\Product\Entity\Product', 'p')
            ->leftJoin ('p.locales', 'l')
            ->where ($query->expr ()->eq ('p.visibility', 1))
            ->orderBy('p.id', 'DESC')
            ->getQuery ()
            ->getArrayResult ();

        return $products;
    }



    public function getProductsCatalog ()
    {

        $collection = SMStatic::getInstance ()->get ('store.collection.productcatalog');
        $params = $this->getParams ();
        $filters = $params->getFilters ();
          foreach ($filters  as $key => $filter){
              if ($filter['field']  == 'product_id'){
                  unset($filters[$key]);
              }
          }

        return $collection->setFilters($filters)
            ->setLimit ($params->getLimit ())
            ->setOffset ($params->getOffset ())
            ->setOrder ($params->getOrder ())
            ->getCollectionWithMetaData ();
    }
    /**
     * @param int $id
     * @return mixed
     */
    public function getItem ($id)
    {
        return $this
            ->collection
            ->setHydrationMode (Query::HYDRATE_ARRAY)
            ->getOne ($id,
                function (&$product) {
                    /** @var \DateTime $dateSpecialStart */
                    $dateSpecialStart = $product ['special_date_start'];

                    if ($dateSpecialStart) {
                        $product ['special_date_start'] = $dateSpecialStart->getTimestamp() * 1000;
                    }

                    /** @var \DateTime $dateSpecialEnd */
                    $dateSpecialEnd = $product ['special_date_end'];

                    if ($dateSpecialEnd) {
                        $product ['special_date_end'] = $dateSpecialEnd->getTimestamp() * 1000;
                    }
                    /** @var \DateTime $dateCreated */
                    $dateCreated = $product ['created_date'];
                    if ($dateCreated) {
                        $product ['created_date'] = $dateCreated->getTimestamp() * 1000;
                    }
                    $tmp_images = $product['images'];



                    foreach ($product['images'] as $key=>$image){


                        $metiaItem =  SMStatic::getInstance ()
                                ->get ('MotoStore\EntityManager')
                            ->getRepository ('MotoStore\Platform\Entity\Media')
                            ->find (array ('id' => $product['images'][$key]['media_id']));

                        if ($metiaItem){
                            $product['images'][$key]['alt'] = $metiaItem->getAlt();
                            $product['images'][$key]['path'] = $metiaItem->getPath();
                            // $product['images'][$key]['path'] = $metiaItem->getPath();
                        } else {
                            unset($product['images'][$key]);
                        }

                    }
                    $product['images'] = array_values($product['images']);


                }
            );
    }

    /**
     * Retrieve Items Collection by Text
     *
     * @param $searchText
     * @param $limit
     * @param $offset
     * @return mixed
     */
    public function search ($searchText, $limit, $offset)
    {
        $collection =  $this
            ->collection
            ->setLimit ($limit)
            ->setOffset ($offset)
            ->setOrder ();

        $query = $collection
            ->getQuery ();

        $query
            ->where (
                $query->expr()->orX (
                    $query->expr ()->like ('l.name', ':search'),
                    $query->expr ()->like ('p.sku', ':search'),
                    $query->expr ()->like ('p.upc', ':search')

                )
            )
            ->setParameter ('search', '%' . $searchText . '%');

        return $collection
            ->getCollectionWithMetaData ();
    }

    /**
     * Retrieve product variant
     *
     * @param int $product_id
     * @param int $finalize_price
     * @return array
     */
    public function getVariants ($product_id = 0, $finalize_price = 0)
    {
        $query = $this
            ->collection
            ->getEntityManager ()
            ->createQueryBuilder ();

        $variants = $query
            ->select(array ('v', 'i', 'vo', 'vv', 'o'))
            ->from ('MotoStore\Product\Entity\Variant', 'v')
            ->leftJoin ('v.image', 'i')
            ->leftJoin ('v.options', 'vo')
            ->leftJoin ('vo.option', 'o')
            ->leftJoin ('vo.value', 'vv')
            ->where ($query->expr ()->eq ('v.product_id', $product_id))
            ->orderBy('v.id', 'ASC')
            ->getQuery ()
            ->getArrayResult ();

        if ($finalize_price && $variants)
        {
            $currency = $this
                ->collection
                ->getEntityManager ()
                ->getRepository ('MotoStore\Settings\Entity\Currency')
                ->getDisplayCurrency ();
            /** @var Product $product */
            $product = $this
                ->collection
                ->getEntityManager ()
                ->getRepository ('MotoStore\Product\Entity\Product')
                ->find ($product_id);

            $variants = array_map (function ($variant) use ($currency, $product) {
                $price = $product->getFinalPrice ();

                $variant ['price'] = floatval($variant ['price']);

                if ($variant ['price'])
                {
                    $price = $variant ['price'];
                }
                $customerDiscount = $this->getCustomerDiscount();
                if($customerDiscount){
                    $price = $price - $price*$customerDiscount/100;
                }

                $variant ['final_price'] = Price::decorate ($price, $currency);

                return $variant;
            }, $variants);
        }
        return $variants;
    }

    private  function getCustomerDiscount()
    {
        $identity  = SMStatic::getInstance ()
            ->get ('store.authentication.service')
            ->getIdentity ();

        if($identity)
        {
            $customer = SMStatic::getInstance ()
                ->get ('MotoStore\EntityManager')
                ->getRepository ('MotoStore\Customer\Entity\Customer')
                ->find ($identity->getId ());
            if ($customer){
                $discount = $customer->getLoyaltyDiscount();
                if($discount && $discount > 0 && $discount < 100){
                    return $discount;
                } else {
                    return false;
                }
            }
        }
        return false;

    }

    /**
     * retrieve product option
     *
     * @param int $product_id
     * @return array
     */
    public function getOptions ($product_id = 0)
    {
        $query = $this
            ->collection
            ->getEntityManager ()
            ->createQueryBuilder ();

        return $query
            ->select(array ('o', 'v'))
            ->from ('MotoStore\Product\Entity\Option', 'o')
            ->leftJoin ('o.values', 'v')
            ->where ($query->expr ()->eq ('o.product_id', $product_id))
            ->orderBy ('o.position', 'ASC')
            ->getQuery ()
            ->getArrayResult ();
    }


    /**
     * Update products visibility
     *
     * @return array
     */
    public function hide ()
    {
        $params = $this->_request->getParams ();

        if (!empty ($params ['id']))
        {
            $product_id     = $params ['id'];

            $em = $this
                ->collection
                ->getEntityManager ();
            /** @var Product $entity */
            $entity = $em->getRepository ('MotoStore\Product\Entity\Product')
                ->find ($product_id);
            $entity
                ->setVisibility(false);

            $em->flush();

        }

    }
    /**
     * Update products visibility
     *
     * @return array
     */
    public function show ()
    {
        $params = $this->_request->getParams ();

        if (!empty ($params ['id']))
        {
            $product_id     = $params ['id'];

            $em = $this
                ->collection
                ->getEntityManager ();

            /** @var Product $entity */
            $entity = $em->getRepository ('MotoStore\Product\Entity\Product')
                ->find ($product_id);
            $entity
                ->setVisibility(true);

            $em->flush();
        }

    }

    /**
     * assignProductToCategory
     */
    public function assignProductToCategory ()
    {
        $params = $this->_request->getParams ();

        if (!empty ($params ['id']) && !empty ($params ['cat_id']))
        {
            $product_id     = $params ['id'];

            $em = $this
                ->collection
                ->getEntityManager ();

            /** @var Product $entity */
            $entity = $em->getRepository ('MotoStore\Product\Entity\Product')
                ->find ($product_id);

            $categories = $this
                ->collection
                ->getEntityManager ()
                ->getRepository ('MotoStore\Product\Entity\Category')
                ->findBy (array('id' => $params ['cat_id']));
            $entity->attachCategories ($categories);
            $em->flush();
        }

    }

     public function duplicate ()
    {
        $params = $this->_request->getParams ();

        if (!empty ($params ['id']))
        {
            $product_id   = $params ['id'];

            $em = $this
                ->collection
                ->getEntityManager ();

            /** @var Product $entity */
            $entity = $em->getRepository ('MotoStore\Product\Entity\Product')
                ->find ($product_id);

            $new_entity = clone $entity;

            $em->persist($new_entity);
            $em->flush();

            /* fix vartiant options */
            $product_variants = $new_entity->getVariants();
            if(count($product_variants) > 0){
                foreach ($product_variants as $variant) {
                    $var_options = $variant->getOptions();
                    if(count($var_options) > 0){
                        foreach ($var_options as $var_option) {
                            $value = $var_option->getValue();
                            $option = $var_option->getOption();
                            $option_name = $option->getName();
                            $product_options = $new_entity->getOptions();
                            if(count($product_options) > 0) {
                                foreach ($product_options as $product_option) {
                                    if($option_name == $product_option->getName()){
                                        $var_option->setOption($product_option);
                                        $newvalues = $product_option->getValues();
                                        if(count($newvalues) > 0) {
                                            foreach ($newvalues as $newvalue) {
                                                if($newvalue->getName() == $value->getName()){
                                                    $var_option->setValue($newvalue);
                                                }
                                            }
                                        }

                                    }
                                }
                            }
                        }
                    }

                }
            }
            $em->persist($new_entity);
            $em->flush();
        }

    }


    public function removeProductFromCategory ()
    {
        $params = $this->_request->getParams ();

        if (!empty ($params ['id']) && !empty ($params ['cat_id']))
        {
            $product_id     = $params ['id'];

            $em = $this
                ->collection
                ->getEntityManager ();

            /** @var Product $entity */
            $entity = $em->getRepository ('MotoStore\Product\Entity\Product')
                ->find ($product_id);

            $entity->removeCategory($entity->getCategories ()->get ($params ['cat_id']));
            $em->flush();
        }

    }



    /**
     * Update products options
     *
     * @return array
     */
    public function saveOptions ()
    {
        $params = $this->_request->getParams ();

        if (!empty ($params ['options']) && !empty ($params ['product_id']))
        {
            $optionsData    = $params ['options'];
            $product_id     = $params ['product_id'];
            
            $em = $this
                ->collection
                ->getEntityManager ();

            /** @var Product $entity */
            $entity = $em->getRepository ('MotoStore\Product\Entity\Product')
                ->find ($product_id);
            
            if ($entity)
            {
                foreach ($optionsData as $position => $option)
                {
                    if (!empty($option ['id']))
                    {
                        $productOption = $entity->getOption ($option ['id']);
                    }
                    else
                    {
                        $productOption = new Option ();
                    }

                    $productOption->setPosition ($position);
                    $productOption->setName ($option ['name']);

                    foreach ($option ['values'] as $value)
                    {
                        if (!empty($value ['id']))
                        {
                            $optionValue = $productOption->getValue($value ['id']);
                        }
                        else
                        {
                            $optionValue = new OptionValue ();
                        }

                        $optionValue->setName ($value ['name']);
                        $optionValue->setOption ($productOption);
                        $productOption
                            ->getValues ()
                            ->add ($optionValue);
                    }

                    $productOption->setProduct ($entity);

                    $em->persist ($productOption);

                    $entity
                        ->getOptions ()
                        ->add ($productOption);
                }

                $em->flush();
                
                return $this->getOptions ($product_id);
            }
        }
        
        return array ();
    }

    /**
     * retrieve product custom_option
     *
     * @param int $product_id
     * @return array
     */
    public function getCustomOptions ($product_id = 0)
    {

        $query = $this
            ->collection
            ->getEntityManager ()
            ->createQueryBuilder ();


        return $query
            ->select(array ('o'))
            ->from ('MotoStore\Product\Entity\CustomOption', 'o')
            ->where ($query->expr ()->eq ('o.product_id', $product_id))
            ->orderBy ('o.position', 'ASC')
            ->getQuery ()
            ->getArrayResult ();
    }


    /**
     * Update products custom_options
     *
     * @return array
     */
    public function saveCustomOptions ()
    {
        $params = $this->_request->getParams ();

        if (!empty ($params ['options']) && !empty ($params ['product_id']))
        {
            $optionsData    = $params ['options'];
            $product_id     = $params ['product_id'];


            $em = $this
                ->collection
                ->getEntityManager ();

            /** @var Product $entity */
            $entity = $em->getRepository ('MotoStore\Product\Entity\Product')
                ->find ($product_id);


            if ($entity)
            {
                foreach ($optionsData as $position => $option)
                {

                    if (!empty($option ['id']))
                    {
                        $productOption = $entity->getCustomOption ($option ['id']);
                    }
                    else
                    {
                        $productOption = new CustomOption ();
                    }


                    $productOption->setPosition ($position);
                    $productOption->setName ($option ['name']);
                    $productOption->setPrice($option ['price']);              

                    $productOption->setProduct ($entity);
                    $productOption->setType ($option ['type']);

                    $em->persist ($productOption);

                    $entity
                        ->getCustomOptions ()
                        ->add ($productOption);


                }

                $em->flush();

                return $this->getCustomOptions ($product_id);
            }
        }

        return array ();
    }
    /**
     * Delete product option
     *
     * @param $product_id
     * @param $option_id
     */
    public function deleteCustomOption ($product_id, $option_id)
    {
        $query = $this
            ->collection
            ->getEntityManager ()
            ->createQueryBuilder ();

        $this
            ->collection
            ->getEntityManager ()
            ->flush ();

        $this
            ->collection
            ->getEntityManager ()
            ->createQueryBuilder ()
            ->delete ('MotoStore\Product\Entity\CustomOption', 'o')
            ->where ($query->expr ()->eq ('o.id', $option_id))
            ->getQuery ()
            ->execute ();
    }

    public function generateVariants($product_id, $remove_all){
        $em = $this
            ->collection
            ->getEntityManager ();

        /** @var Product $product_entity */
        $product_entity = $em->getRepository ('MotoStore\Product\Entity\Product')
            ->find ($product_id);

        if($remove_all){
            $product_variants = $product_entity->getVariants();
            if(count($product_variants) > 0){
                foreach ($product_variants as $variant) {
                    $em->remove ($variant);
                    $em->flush();
                }
            }
        }
        $all_options = $this->getOptions($product_id);
        $opt_ids = [];
        $values_arrays= [];
        foreach ($all_options as $k=>$option){
            $opt_ids [$k]= $option['id'];
            $values_arrays [$k]= $option['values'];

        }

        $variants = $this->generateCombinationsArray($values_arrays);
        foreach ($variants as $single_selected_option){
            $variant_type = new VariantType();
            $variant_type
                ->setEntityManager ($em)
                ->init ();
            $variant_entity = new Variant();
            $variant_data = array(
                'sku' => "",
                'price' => $product_entity->getPrice(),
                'product_id' => $product_id,
                'quantity' => $product_entity->getQuantity(),
            );
            $options_selected = [];
            foreach ($single_selected_option as $single_opt_value){
                $tmp_selected = array(
                    'id'=> $single_opt_value['option_id'],
                    'selectedValue' =>$single_opt_value
                );
                $options_selected [] = $tmp_selected;
            }
            $variant_data['options'] = $options_selected;
            $variant_type->handle ($variant_data, $variant_entity);
        }

    }

    public function generateCombinationsArray ($arrays, $global_n=-1, $count=false, $weight=false)
    {
        if ($global_n == -1) {
            $arrays = array_values($arrays);
            $count = count($arrays);
            $weight = array_fill(-1, $count+1, 1);
            $Q = 1;
            foreach ($arrays as $i=>$array) {
                $size = count($array);
                $Q = $Q * $size;
                $weight[$i] = $weight[$i-1] * $size;
            }

            $result = array();
            for ($n=0; $n<$Q; $n++)
                $result[] = $this->generateCombinationsArray($arrays, $n, $count, $weight);

            return $result;
        }
        else {

            $SostArr = array_fill(0, $count, 0);

            $oldN = $global_n;

            for ($i=$count-1; $i>=0; $i--)
            {
                $SostArr[$i] = floor( $global_n/$weight[$i-1] );
                $global_n = $global_n - $SostArr[$i] * $weight[$i-1];
            }

            $result = array();
            for ($i=0; $i<$count; $i++)
                $result[$i] = $arrays[$i][ $SostArr[$i] ];
            return $result;
        }

    }
    /**
     * @param int $variant_id
     */
    public function deleteVariant ($variant_id = 0)
    {
        $em = $this
            ->collection
            ->getEntityManager ();

        /** @var Variant $variant */
        $variant = $em->find('MotoStore\Product\Entity\Variant', $variant_id);

        if ($variant)
        {
            $em->remove ($variant);
            $em->flush();
        }
    }

    /**
     * @param $product_id
     * @param int $value_id
     */
    public function deleteOptionValue ($product_id, $value_id = 0)
    {
        $query = $this
            ->collection
            ->getEntityManager ()
            ->createQueryBuilder ();

        $query
            ->delete ('MotoStore\Product\Entity\Variant\VariantOption', 'v')
            ->where ($query->expr ()->eq ('v.value_id', $value_id))
            ->getQuery ()
            ->execute ();

        $query
            ->delete ('MotoStore\Product\Entity\OptionValue', 'v')
            ->where ($query->expr ()->eq ('v.id', $value_id))
            ->getQuery ()
            ->execute ();

        $this->removeEmptyVariants ($product_id);


    }


    /**
     * Delete product option
     *
     * @param $product_id
     * @param $option_id
     */
    public function deleteOption ($product_id, $option_id)
    {
        $query = $this
            ->collection
            ->getEntityManager ()
            ->createQueryBuilder ();

        $emptyVariants = $query
            ->select (array('v'))
            ->from ('MotoStore\Product\Entity\Variant', 'v')
            ->join ('v.options', 'vo')
            ->where ($query->expr ()->eq ('v.product_id', $product_id))
            ->andWhere ($query->expr ()->eq ('vo.option_id', $option_id))
            ->getQuery ()
            ->getResult ();

        foreach ($emptyVariants as $variant)
        {
            $this->collection->getEntityManager ()->remove ($variant);
        }

        $this
            ->collection
            ->getEntityManager ()
            ->flush ();

        $this
            ->collection
            ->getEntityManager ()
            ->createQueryBuilder ()
            ->delete ('MotoStore\Product\Entity\Option', 'o')
            ->where ($query->expr ()->eq ('o.id', $option_id))
            ->getQuery ()
            ->execute ();
    }



    /**
     * Remove variants without combination
     * @param $product_id
     */
    private function removeEmptyVariants ($product_id)
    {
        $query = $this
            ->collection
            ->getEntityManager ()
            ->createQueryBuilder ();

        $optionsCount = count ($this->getOptions ($product_id));

        $emptyVariants = $query
            ->select (array('v'))
            ->from ('MotoStore\Product\Entity\Variant', 'v')
            ->join ('v.options', 'vo')
            ->where ($query->expr ()->eq ('v.product_id', $product_id))
            ->having('count(vo.id) < :options')
            ->groupBy ('v.id')
            ->setParameter ('options', $optionsCount)
            ->getQuery ()
            ->getResult ();

        foreach ($emptyVariants as $variant)
        {
            $this->collection->getEntityManager ()->remove ($variant);
        }

        $this
            ->collection
            ->getEntityManager ()
            ->flush ();
    }

    /**
     * @param $ids
     * @param $category_id
     * @return bool
     */
    public function updatePosition ($ids, $category_id)
    {
        /** @var ProductRepository $repository */
        $repository = $this
            ->collection
            ->getEntityManager ()
            ->getRepository ('MotoStore\Product\Entity\Product');


        foreach ($ids as $position => $productID)
        {
            $repository
                ->updatePosition ($productID, $category_id, $position);
        }
    }

    /**
     * Export Collection
     * @return array
     */

    public function export ()
    {

        $params = $this->_request->getParams ();
        $exportdir = Moto\System::getAbsolutePath($this->globalConfig['settings']['exportPath']);
        $exported =  $this->getPrefix () . '-' . date('Y_m_d_H_i_s') . '.csv';
        $adapter = new Export ($exportdir . $exported, $this->collection, $params);
        $adapter
            ->export ();
         return array (
             'exported'=>  Moto\System::getAbsoluteUrl($this->globalConfig['settings']['exportPath'] . $exported)
         );
     }


    public function import ($source)
    {
        $resource_file  = join ('/', array(\Moto\System::getAbsolutePath('@userUploads'), $source));
        $adapter        = new Import ($this->collection->getEntityManager ());
        //error_log(json_encode(array($resource_file)));
        if (file_exists ($resource_file) && is_readable ($resource_file))
        {
            //error_log('file_exists');
            $file = fopen ($resource_file, 'r');

            $adapter->import ($file);
        }

        return $adapter->getResult ();
    }

    public function analyzeImportFile($source,$settings){
        $resource_file  = join ('/', array(\Moto\System::getAbsolutePath('@userUploads'), $source));
        $adapter        = new Import ($this->collection->getEntityManager ());
        //error_log(json_encode(array($resource_file)));
        $result = array();
        if (file_exists ($resource_file) && is_readable ($resource_file))
        {
            //error_log('file_exists');
            $file = fopen ($resource_file, 'r');

            $result = $adapter->analyzeFile ($file,$settings);
        }

        return $result;
    }



    public function importLine ($line, $prev_id, $settings)
    {
        $adapter  = new Import ($this->collection->getEntityManager ());
        return $adapter->importLine ($line, $settings, $prev_id);
    }


    public function getCurrencyToken(){

        $currency = $this
            ->collection
            ->getEntityManager ()
            ->getRepository ('MotoStore\Settings\Entity\Currency')
            ->getDisplayCurrency ();

        return  $currency->getToken ();
    }
    public function getCurrency(){
        $currency = $this
            ->collection
            ->getEntityManager ()
            ->getRepository ('MotoStore\Settings\Entity\Currency')
            ->getDisplayCurrency ();

        return  array(
            'token' => $currency->getToken (),
            'decimalPlaces' => $currency->getDecimalPlaces (),
            'decimalPoint' => $currency->getDecimalPoint (),
            'thousandsSeparator' => $currency->getThousandsSeparator (),
            'location' => $currency->getLocation(),
            'exchangeRate' => $currency->getExchangeRate(),
            'spacer' => $currency->getSpacer()
        );
    }

    public function getProductByBrand ($brand_id)
    {
        $repository = $this
            ->collection
            ->getEntityManager ()
            ->getRepository ('MotoStore\Product\Entity\Product');



        return $repository->getProductByBrandId($brand_id);
    }


    public function removeProductBrand($id){
        $params = $this->_request->getParams ();

        if (!empty ($params ['id']) )
        {
            $product_id     = $params ['id'];

            $em = $this
                ->collection
                ->getEntityManager ();

            /** @var Product $entity */
            $entity = $em->getRepository ('MotoStore\Product\Entity\Product')
                ->find ($product_id);

            $entity->removeBrands();
            $em->flush();
        }
    }

}