<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Service;

use Doctrine\ORM\Query;
use Moto\Json\Server\Exception;
use Moto\Service\AbstractService;
use MotoStore\Collection\Collection;
use MotoStore\Collection\CollectionHelper;
use MotoStore\Platform\Form\FormTypeAbstract;
use MotoStore\Product\Entity\Product;
use Zend\EventManager\EventManager;
use Zend\ServiceManager\ServiceLocatorInterface;


abstract class StoreServiceAbstract extends AbstractService implements ServiceReadableInterface
{
    const GLOBAL_RESOURCES_PREFIX = 'store_';

    const EVENT_OPERATION_UPDATE        = 'update';
    const EVENT_OPERATION_REMOVE        = 'remove';

    /**
     * Global resource map
     * @var array
     */
    protected $_resourcePrivilegesMap = array (
        'getCollection' => 'get',
        'getItem' => 'get',
    );

    /**
     * @var Collection
     */
    protected $collection;

    /**
     * @var EventManager
     */
    protected $eventManager;

    protected $globalConfig;

    /**
     * @param ServiceLocatorInterface $serviceLocator
     */
    public function __construct (ServiceLocatorInterface $serviceLocator)
    {
        $this->collection   = $serviceLocator->get ($this->getCollectionName ());
        $this->eventManager = $serviceLocator->get ('store.eventmanager');
        $this->globalConfig = $serviceLocator->get ('store.config');
        $this->applyResourcePrefix ();
    }

    /**
     * Prefix of ACL Resource
     *
     * @return mixed
     */
    protected abstract function getPrefix ();

    /**
     * Collection Name
     *
     * @return string
     */
    protected abstract function getCollectionName ();

    /**
     * Apply ACL Resource Prefix
     */
    protected function applyResourcePrefix ()
    {
        $this->_resourceName = self::GLOBAL_RESOURCES_PREFIX . $this->getPrefix ();
    }

    /**
     * Collection Params
     *
     * @return CollectionHelper
     */
    protected function getParams ()
    {
        return new CollectionHelper ($this->_request->getParams());
    }

    /**
     * Entity update request handler
     *
     * @param FormTypeAbstract $formType
     * @param $object
     * @return object
     * @throws Exception
     */
    protected function handleUpdateRequest (FormTypeAbstract $formType, $object)
    {
        $params = $this->_request->getParams ();

        if (!empty ($params ['id']))
        {
            $entity = $this
                ->collection
                ->setHydrationMode (Query::HYDRATE_OBJECT)
                ->getOne ($params ['id']);

            if ($entity)
            {
                $object = $entity;
            }
        }

        $formType
            ->setEntityManager ($this->collection->getEntityManager ())
            ->init ();

        $this
            ->eventManager
            ->trigger ($this->getEventName (self::EVENT_OPERATION_UPDATE), null, $object);


        if ($formType->handle ($params, $object))
        {
            return $this->getItem ($object->getId ());
        }

        throw new Exception ('ERROR.BAD_REQUEST', 0, $this->handleFormErrors ($formType->getMessages ()));
    }

    /**
     * Handle Collection
     * @param callable $callback
     * @return array
     */
    protected function handleCollectionRequest ($callback = null)
    {
        $params = $this->getParams ();

        return $this
            ->collection
            ->setFilters($params->getFilters ())
            ->setLimit ($params->getLimit ())
            ->setOffset ($params->getOffset ())
            ->setOrder ($params->getOrder ())
            ->getCollectionWithMetaData ($callback);
    }

    /**
     * @param $id
     */
    protected function handleDeleteRequest ($id)
    {
        $entity = $this
            ->collection
            ->setHydrationMode (Query::HYDRATE_OBJECT)
            ->getOne ($id);

        if ($entity)
        {
            $this
                ->eventManager
                ->trigger ($this->getEventName (self::EVENT_OPERATION_REMOVE), null, $entity);

            $this
                ->collection
                ->getEntityManager ()
                ->remove ($entity);

            $this
                ->collection
                ->getEntityManager ()->flush ();
        }
    }

    /**
     * Retrieve Collection By Params
     * @return mixed
     * @throws Exception
     */
    public function getCollection ()
    {
        throw new Exception ('ERROR.BAD_REQUEST');
    }

    /**
     * @param int $id
     * @return mixed
     * @throws Exception
     */
    public function getItem ($id)
    {
        throw new Exception ('ERROR.BAD_REQUEST');
    }

    /**
     * Get collection event name
     *
     * @return string
     */
    protected function getEventName ($operation)
    {
        return 'store.event.' . $this->getPrefix () . '.' . $operation;
    }


    /**
     * Return form messages
     * @param $messages
     * @return array
     */
    private function handleFormErrors ($messages)
    {
        return array_map ('array_keys', $messages);
    }


}