<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Collection;

use Doctrine\ORM\Query;
use Doctrine\ORM\QueryBuilder;
use Doctrine\ORM\Tools\Pagination\Paginator;
use MotoStore\Collection\Query\Designer;
use MotoStore\Collection\Query\Param;

class Collection implements CollectionInterface
{
    /**
     * @var QueryBuilder
     */
    protected $query;

    /**
     * @var Designer
     */
    protected $queryDesigner;

    /**
     * @var array
     */
    protected $propertiesMap = array ();

    /**
     * @var int
     */
    protected $hydrationMode = Query::HYDRATE_ARRAY;

    /**
     * @var Paginator
     */
    protected $pagination;

    /**
     * @return QueryBuilder
     */
    public function getQuery ()
    {
        return $this->query;
    }

    /**
     * @param QueryBuilder $query
     */
    public function setQuery ($query)
    {
        $this->query = $query;
    }

    /**
     * Set hydration mode
     *
     * @param $mode
     * @return $this
     */
    public function setHydrationMode ($mode)
    {
        $this->hydrationMode = $mode;

        return $this;
    }

    /**
     * @return array
     */
    public function getPropertiesMap ()
    {
        return $this->propertiesMap;
    }

    /**
     * @param array $propertiesMap
     */
    public function setPropertiesMap ($propertiesMap)
    {
        $this->propertiesMap = $propertiesMap;
    }

    /**
     * @param array $filters
     * @return Collection
     */
    public function setFilters (array $filters)
    {
        if ($filters)
        {
            $designer = $this->getQueryDesigner ();

            $designer->resetParams ();

            array_map (
                function ($filter) use ($designer)
                {
                    if ($filter instanceof Param)
                    {
                        $designer->addParam ($filter);
                    }
                    else
                    {
                        $designer->addFilter ($filter ['field'], $filter ['operator'], $filter ['value']);
                    }
                },
                $filters
            );

            $this->queryDesigner->applyParams ();
        }

        return $this;
    }

    /**
     * Retrieve entity collection
     *
     * @param callable $callback
     * @return array
     */
    public function getCollection ($callback = null)
    {
        if ($callback !== null)
        {
            return array_map ($callback, $this->getQueryResult ());
        }

        return $this->getQueryResult ();
    }

    /**
     * Fetch one element
     *
     * @param $id
     * @param callable $callback
     * @return array
     */
    public function getOne ($id, $callback = null)
    {
        $root = $this
            ->getAlias ($this->getQuery ());

        $this
            ->getQuery ()
            ->andWhere ($this->getQuery ()->expr ()->eq ($root [0] . '.id', $id));

        if ($result = $this->getQuery ()->getQuery ()->getResult ($this->hydrationMode))
        {
            if ($callback !== null)
            {
                $callback ($result [0]);
            }
            return $result [0];
        }
        return array ();
    }

    /**
     * Get result from query
     *
     * @return array
     */
    protected function getQueryResult ()
    {
        $query = $this
            ->getQuery ()
            ->getQuery ()
            ->setHydrationMode ($this->hydrationMode);

        $this->pagination = new Paginator ($query);
        $this->pagination->setUseOutputWalkers(false);

        return (array) $this
            ->pagination
            ->getIterator ();
    }

    /**
     * Query filters designer
     *
     * @return Designer
     */
    public function getQueryDesigner ()
    {
        if (!$this->queryDesigner)
        {
            $this->queryDesigner = new Designer ($this->getQuery (), $this->getPropertiesMap ());
        }
        return $this->queryDesigner;
    }

    /**
     * Set result count
     *
     * @param $limit
     * @return Collection
     */
    public function setLimit ($limit)
    {
        $this->getQuery ()->setMaxResults ($limit);

        return $this;
    }

    /**
     * Set Result Offset
     *
     * @param $offset
     * @return Collection
     */
    public function setOffset ($offset)
    {
        $this->getQuery ()->setFirstResult ($offset);

        return $this;
    }

    /**
     * @param array $order
     * @return Collection
     */
    public function setOrder ($order = array ())
    {
        if ($order)
        {
          foreach ($order as $orderOptions)
            {
                if ($orderOptions)
                {
                    $this
                        ->getQueryDesigner ()->resetOrderParams();
                    $this
                        ->getQueryDesigner ()
                        ->addOrderParams ($orderOptions ['order'], $orderOptions ['direction']);

                }
            }
        }
        return $this;
    }


    /**
     * Retrieve collection with metadata
     * Callback should be used for every element of a collection
     *
     * @param callable $callback
     * @return array
     */
    public function getCollectionWithMetaData ($callback = null)
    {
        return array(
            'collection'    => $this->getCollection ($callback),
            'totalCount'    => $this->getCollectionCount (),
        );
    }

    /**
     * Get total count
     *
     * @return mixed
     */
    public function getCollectionCount ()
    {
        return $this
            ->pagination
            ->count ();
    }

    /**
     * Retrieve alias of main entity
     *
     * @param QueryBuilder $query
     * @return null
     */
    protected function getAlias (QueryBuilder $query)
    {
        return $query->getRootAliases();
    }

    /**
     * @return \Doctrine\ORM\EntityManager
     */
    public function getEntityManager ()
    {
        return $this->getQuery ()->getEntityManager ();
    }
}