<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Content\Presenter;


use MotoStore\Collection\Collection;
use MotoStore\Collection\Query\Param;
use MotoStore\Content\Form\EditAccountType;
use MotoStore\Content\Form\EditPersonalType;
use MotoStore\Content\Helper\Link;
use MotoStore\Content\Helper\Pagination;
use MotoStore\Content\Helper\Translator;
use MotoStore\Customer\Entity\Customer;
use MotoStore\Order\Entity\Order;
use Zend\Authentication\AuthenticationService;
use MotoStore\Content\Helper\PDFGenerator;
use MotoStore\Settings\Entity\Option;
use MotoStore\Content\Content\DI\SMStatic;

class AccountPresenter extends AbstractPresenter
{
    /**
     * @var array
     */
    protected $errors = array ();

    /**
     * Request data
     * @var array
     */
    protected $data = array ();

    /**
     * @var AuthenticationService
     */
    protected $identityService;

    /**
     * @var Customer
     */
    protected $customer;

    /**
     * @var
     */
    protected $section;

    /** @var  Pagination */
    private $pagination;

    /**
     * @param $uri
     */
    public function __construct ($uri)
    {
        $this->section = $uri;

        $this->identityService = $this
            ->getServiceLocator ()
            ->get ('store.authentication.service');
    }

    /**
     * Create Pagination Helper
     *
     * @return Pagination
     */
    protected function getPagination ()
    {
        if (!$this->pagination)
        {
            $this->pagination = new Pagination ($this->getRequest ());
        }
        return $this->pagination;
    }

    /**
     * @return Customer
     */
    public function getIdentity ()
    {
        return $this->identityService->getIdentity ();
    }

    /**
     * Present Page Data
     *
     * @return array
     */
    public function present ()
    {
        $optRepository = $this
            ->getServiceLocator ()
            ->get ('MotoStore\EntityManager')
            ->getRepository ('MotoStore\Settings\Entity\Option');

        return array (
            'customer'      => $this->getCustomer (),
            'custom_information'      => $this->getCustomerCustomInformation (),
            'orders'        => $this->presentAccountOrders (),
            'currency'      => $this->presentCurrency (false),
            'countries'     => $this->presentCountries (),
            'states'        => $this->presentStates (),
            'errors'        => $this->errors,
            'perPage'       => $this->getPagination ()->getLimit (),
            'pages'         => $this->getPagination ()->getPagesCount (),
            'section'       => $this->section,
            // Present request data
            'data'          => $this->data,
            'disable_prices_on_cart_and_checkout' => $this->isPricesEnabled()
        );
    }
    protected function isPricesEnabled ()
    {
        $optRepository = clone SMStatic::getInstance()
            ->get ('MotoStore\EntityManager')
            ->getRepository ('MotoStore\Settings\Entity\Option');

        $catalog_mode = $optRepository ->getOption(Option::OPTION_STORE_SETTINGS_PRODUCT, 'catalog_mode');
        return !!$catalog_mode;
    }
    /**
     * Save Account Details
     * Handle POST request
     *
     */
    public function handle ()
    {
        $this->data = $this->getRequest ()->getPost ();
        
        if  ($this->getRequest ()->getPost ('type') == 'generate_invoice'){
            $filelink = PDFGenerator::generateInvoice($this->getRequest ()->getPost ('order_id'));
            header ("Location: {$filelink}");
            die ();
        }

        $type = $this->getRequest ()->getPost ('type') == 'chp'
            ? new EditPersonalType ()
            : new EditAccountType ();

        $type
            ->setEntityManager ($this->getServiceLocator ()->get ('MotoStore\EntityManager'))
            ->init ();
        

        if (!$type->handle ($this->getRequest ()->getPost (), $this->getCustomer ()))
        {
            $this->errors = $type->getMessages ();
        }
        else
        {
            // update data in session storage
            $this->identityService->getStorage ()
                ->write ($this->getCustomer ());

            Link::redirect ('account', 'edit');
        }
    }

    /**
     * Present Customer Orders
     *
     * @return array
     */
    public function presentAccountOrders ()
    {
        if (!$this->getCustomer ())
        {
            return array();
        }

        /**
         * @var Collection $collection
         */
        $collection = $this
            ->getServiceLocator ()
            ->get ('store.collection.order');

        $collection
            ->setFilters (array (
                new Param ('customer_id', Param::OPERATOR_EQ, $this->getCustomer ()->getId ())
            ))
            ->setOffset ($this->getPagination ()->getFirstResult ())
            ->setLimit ($this->getPagination ()->getLimit ())
            ->setOrder (array(
                array (
                    'order'     => 'id',
                    'direction' => 'DESC'
                )
            ));

        $result = $collection
            ->getCollectionWithMetaData (function ($order) {
                $order['presented_currency'] = $this->presentCurrency (false, $order['currency']['id']);
                $order ['status_text'] = str_replace('STORE.ORDER.','', Order::getStatusName ($order ['status']));
                return $order;
            });

        $this
            ->getPagination ()
            ->setResultCount ($result ['totalCount']);

        return $result ['collection'];
    }

    /**
     * @return Customer
     */
    protected function getCustomer ()
    {
        if (!$this->customer && $this->getIdentity ())
        {
            $this->customer = $this
                ->getServiceLocator ()
                ->get ('MotoStore\EntityManager')
                ->find ('MotoStore\Customer\Entity\Customer', $this->getIdentity ()->getId ());
        }
        return $this->customer;
    }
    /**
     * @return Customer
     */
    protected function getCustomerCustomInformation ()
    {
        if (!$this->customer && $this->getIdentity ())
        {
            $this->customer = $this
                ->getServiceLocator ()
                ->get ('MotoStore\EntityManager')
                ->find ('MotoStore\Customer\Entity\Customer', $this->getIdentity ()->getId ());
        }
        $info = ($this->customer)?$this->customer->getCustomInformation():'';
        return $info;
    }

    /**
     * Present country
     * @return array
     */
    protected function presentCountries ()
    {
        /**
         * @var Collection $collection
         */
        $collection = $this
            ->getServiceLocator ()
            ->get ('store.collection.country');

        return $collection->getCollection ();
    }


    /**
     * Present Store States
     * @return array
     */
    protected function presentStates ()
    {
        /**
         * @var Collection $collection
         */
        $collection = $this
            ->getServiceLocator ()
            ->get ('store.collection.state');

        return $collection->getCollection ();
    }
}