<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Customer\Service;

use Doctrine\ORM\Query;
use MotoStore\Content\Helper\Link;
use MotoStore\Customer\Entity\Customer;
use MotoStore\Customer\Form\CustomerType;
use MotoStore\Order\Cart\Storage;
use MotoStore\Service\ServiceEditableInterface;
use MotoStore\Service\ServiceReadableInterface;
use MotoStore\Service\ServiceSearchableInterface;
use MotoStore\Service\StoreServiceAbstract;
use Zend\Authentication\AuthenticationService;
use Zend\ServiceManager\ServiceLocatorInterface;
use MotoStore\Mailer\Entity\Template;
use MotoStore\Platform\Auth\DoctrineAdapter;

class CustomerServiceClass extends StoreServiceAbstract
    implements ServiceReadableInterface, ServiceEditableInterface, ServiceSearchableInterface
{
    /**
     * @var AuthenticationService
     */
    protected $authentication;
    protected $mailer;

    protected $_resourcePrivilegesMap = array (
        'auth' => 'get',
        'register' => 'get',
        'logout' => 'get',
    );

    public function __construct(ServiceLocatorInterface $serviceLocator)
    {
        $this->authentication = $serviceLocator->get ('store.authentication.service');
        $this->mailer = $serviceLocator->get ('store.mailer');

        parent::__construct($serviceLocator);
    }

    /**
     * Prefix of Resource
     *
     * @return mixed
     */
    protected function getPrefix ()
    {
        return 'customer';
    }

    /**
     * Collection Name
     *
     * @return string
     */
    protected function getCollectionName ()
    {
        return 'store.collection.customer';
    }

    /**
     * Save Resource
     *
     * @return mixed
     */
    public function save ()
    {
        return $this
            ->handleUpdateRequest (new CustomerType (), new Customer ());
    }

    /**
     * Remove Resource
     *
     * @param $id
     * @return mixed
     */
    public function delete ($id)
    {
        $this
            ->handleDeleteRequest ($id);
    }

    /**
     * Retrieve Collection By Params
     *
     * @return mixed
     */
    public function getCollection ()
    {
        return $this
            ->handleCollectionRequest ();
    }

    /**
     * @param int $id
     * @return mixed
     */
    public function getItem ($id)
    {
        return $this
            ->collection
            ->setHydrationMode (Query::HYDRATE_ARRAY)
            ->getOne ($id);
    }

    /**
     * Retrieve Items Collection by Text
     *
     * @param $searchText
     * @param $limit
     * @param $offset
     * @return mixed
     */
    public function search ($searchText, $limit, $offset)
    {
        $collection =  $this
            ->collection
            ->setLimit ($limit)
            ->setOffset ($offset)
            ->setOrder ();

        $query = $collection
             ->getQuery ();

        $query
            ->where (
                // search in full name
                $query->expr()->orX (
                    $query->expr ()->like (
                        $query->expr ()->concat (
                            'c.first_name',
                            $query->expr ()->concat ($query->expr ()->literal (' '), 'c.last_name')
                        ),
                        ':search'
                    ),
                    $query->expr()->like ('c.email', ':search'),
                    $query->expr()->like ('c.contact_phone', ':search')

                )
            )
            ->setParameter ('search', '%' . $searchText . '%');

        return $collection
            ->getCollectionWithMetaData ();
    }

    /**
     * @return bool
     */
    public function auth ()
    {
        $this
            ->authentication
            ->getAdapter ()
            ->setEmail ($this->getParams ()->get ('email'))
            ->setPassword ($this->getParams ()->get ('password', ''));

        $result = $this
            ->authentication
            ->authenticate ();

        if ($result->isValid ())
        {
            return array (
                'result' => true,
                'target' => Link::storeLink ()
            );
        }
        return array (
            'result' => false,
        );
    }

    /**
     * Register new customer
     *
     * @return bool
     */
    public function register ()
    {
        $password  = $this
            ->getParams ()
            ->get ('password');

        $confirmation = $this
            ->getParams ()
            ->get ('confirm_password');

        $email = $this
            ->getParams ()
            ->get ('email');

        if ($password === $confirmation)
        {
            $customer = new Customer ();
            $customer->setEmail ($email);
            $customer->setPassword ($this->authentication->getAdapter ()->createPasswordHash ($email, $password));
            $customer->setActive (1);
            $customer->setFirstName ($this->getParams ()->get ('first_name'));
            $customer->setLastName ($this->getParams ()->get ('last_name'));

            $this
                ->collection
                ->getEntityManager ()
                ->persist ($customer);
            $this
                ->collection
                ->getEntityManager ()
                ->flush ();

            // Login User
            $verification = $this->auth ();

            if ($verification ['result'])
            {
                return array (
                    'result' => true,
                    'target' => Link::storeLink ('account')
                );
            }
        }
        return array (
            'result' => false,
        );
    }

    /**
     * Logout
     */
    public function logout ()
    {
        $this
            ->authentication
            ->clearIdentity ();

        $cart = new Storage ();
        $cart
            ->clear ();
    }

    public function current ()
    {
        return $this
            ->authentication
            ->getIdentity ();
    }
    public function setBtob ()
    {
        $params = $this->_request->getParams ();

        if (!empty ($params ['id']))
        {
            $customer_id     = $params ['id'];
            $btob = false;
            if ($params ['btob'] == 'true'){
                $btob = true;
            }
            $em = $this
                ->collection
                ->getEntityManager ();
            /** @var Customer $entity */
            $entity = $em->getRepository ('MotoStore\Customer\Entity\Customer')
                ->find ($customer_id);
            $entity
                ->setBtob($btob);

            $em->flush();

        }

    }

    public function setActive ()
    {
        $params = $this->_request->getParams ();

        if (!empty ($params ['id'])) {

            $customer_id = $params ['id'];
            $active = false;
            if ($params ['active'] == 'true') {
                $active = true;
            }

            $em = $this
                ->collection
                ->getEntityManager();
            /** @var Customer $entity */
            $entity = $em->getRepository('MotoStore\Customer\Entity\Customer')
                ->find($customer_id);
            $entity
                ->setActive($active);

            $em->flush();

        }

    }

    public function resetPassword(){
        $params = $this->_request->getParams ();

        if (!empty ($params ['id'])) {
            $customer_id = $params ['id'];

            $em = $this
                ->collection
                ->getEntityManager();

            $customer = $em->getRepository('MotoStore\Customer\Entity\Customer')
                ->find($customer_id);

            if ($customer)
            {
                $password = DoctrineAdapter::generatePassword ();
                if (!empty ($params ['password'])){
                    $password = $params ['password'];
                }
                $customer
                    ->setPassword (DoctrineAdapter::createPasswordHash ($customer->getEmail (), $password));

                $em->merge ($customer);
                $em->flush ();

                $this->mailer->sendTemplate (
                        array ($customer->getEmail() => $customer->getFullName ()),
                        Template::TEMPLATE_TARGET_FORGOT_PASSWORD,
                        array (
                            'customer' => $customer,
                            'password' => $password
                        )
                    );

                $success = true;
            }
        }
    }
}