<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Order\Service;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\Query;
use MotoStore\Content\Helper\Link;
use MotoStore\Customer\Entity\Customer;
use MotoStore\Order\Cart\Storage;
use MotoStore\Product\Entity\Product;
use MotoStore\Service\StoreServiceAbstract;
use MotoStore\Settings\Entity\Discount;
use MotoStore\Settings\Entity\Option;
use MotoStore\Settings\Repository\OptionRepository;
use Zend\Authentication\AuthenticationService;
use Zend\ServiceManager\ServiceLocatorInterface;
use Zend\EventManager\EventManager;
use MotoStore\Content\Content\DI\SMStatic;
use Zend\Filter;
use Zend\Validator;
use Zend\Stdlib\ErrorHandler;
use Moto;

class CartServiceClass extends StoreServiceAbstract
{
    /**
     * @var AuthenticationService
     */
    protected $authentication;

    /**
     * @var EntityManager
     */
    protected $entityManager;

    /** @var  Storage */
    protected $cartStorage;
    
    /** @var  OptionRepository */
    protected $optionRepository;
    
    /**
     * @param ServiceLocatorInterface $serviceLocator
     */
    public function __construct (ServiceLocatorInterface $serviceLocator)
    {
        $this->authentication   = $serviceLocator->get ('store.authentication.service');
        $this->entityManager    = $serviceLocator->get ('MotoStore\EntityManager');
        $this->optionRepository = $this->entityManager->getRepository ('MotoStore\Settings\Entity\Option');
    }

    /**
     * Prefix of Resource
     *
     * @return mixed
     */
    protected function getPrefix ()
    {
        return 'cart';
    }

    /**
     * Collection Name
     *
     * @return string
     */
    protected function getCollectionName ()
    {
    }


    /**
     * Add item to cart
     */
    public function addItem ()
    {
        $params = $this
            ->_request
            ->getParams ();
        
        $product_id = !empty ($params ['product_id']) ? $params ['product_id'] : 0;
        $quantity   = !empty ($params ['quantity']) ? $params ['quantity'] : 0;
        $variant_id = isset ($params ['variant_id']) ? $params ['variant_id'] : 0;
        $custom_options = isset ($params ['custom_options']) ? $params ['custom_options'] : 0;

        /** @var Product $product */
        $product = $this->entityManager
            ->getRepository ('MotoStore\Product\Entity\Product')
            ->find ($product_id);
        
        if ($product)
        {

            if ($this->checkProductInventory ($product, $quantity, $variant_id))
            {
                $this
                    ->getCartStorage ()
                    ->addItem ($product, $quantity, $variant_id, $custom_options);
            }
        }
    }

    /**
     * Add item to compare
     */
    public function addCompareItem ()
    {
        $params = $this
            ->_request
            ->getParams ();

        $product_id = !empty ($params ['product_id']) ? $params ['product_id'] : 0;

        /** @var Product $product */
        $product = $this->entityManager
            ->getRepository ('MotoStore\Product\Entity\Product')
            ->find ($product_id);
        if ($product)
        {
            $this
                ->getCartStorage ()
                ->addCompareItem($product);
        }
    }

    /**
     * @param Product $product
     * @param $quantity
     * @return bool
     */
    private function checkProductInventory (Product $product, $quantity, $variant_id)
    {
        $trackInventory = $this
            ->optionRepository
            ->getOption (Option::OPTION_STORE_SETTINGS_PRODUCT, 'track_product_inventory', false);

        $useNegativeInventory = $this
            ->optionRepository
            ->getOption (Option::OPTION_STORE_SETTINGS_PRODUCT, 'allow_negative_inventory', false);

        if ($trackInventory)
        {
            if ($variant_id)
            {
                $variant = $product->getVariant ($variant_id);

                if (!$variant || ($variant->getQuantity() - $quantity < 0))
                {
                    return false;
                }
            }

            if ($product->getQuantity () - $quantity < 0)
            {
                if (!$useNegativeInventory)
                {
                    return false;
                }
            }
        }
        return true;
    }
    
    
    /**
     * Update cart Item
     */
    public function updateItem ()
    {
        $params = $this
            ->_request
            ->getParams ();

        $product_id_combined = !empty ($params ['product_id']) ? $params ['product_id'] : '';
        $quantity   = !empty ($params ['quantity']) ? $params ['quantity'] : 0;
        
        $ids = explode('_', $product_id_combined);
        $id = !empty($ids [0]) ? $ids [0] : 0;
        $idv = !empty($ids [1]) ? $ids [1] : 0;

        /** @var Product $product */
        $product = $this->entityManager
            ->getRepository ('MotoStore\Product\Entity\Product')
            ->find ($id);

        if ($product)
        {
            if ($this->checkProductInventory ($product, $quantity, $idv))
            {
                $this
                    ->getCartStorage ()
                    ->updateItem ($product_id_combined, $quantity);
            }

        }


    }

    /**
     * Remove cart Item
     */
    public function removeItem ()
    {
        $params = $this
            ->_request
            ->getParams ();

        $this
            ->getCartStorage ()
            ->removeItem ($params ['product_id']);
    }
    /**
     * Remove compare Item
     */
    public function removeCompareItem ()
    {
        $params = $this
            ->_request
            ->getParams ();

        $this
            ->getCartStorage ()
            ->removeCompareItem($params ['product_id']);
    }


    public function setShippingMethod ()
    {
        $params = $this
            ->_request
            ->getParams ();

        $this
            ->getCartStorage ()
            ->setShippingMethod ($params ['id']);
        return true;
    }

    /**
     * Get cart and relative data
     * CartWidget
     *
     * @return array
     */
    public function getCart ()
    {
        /** @var Customer $identity */
        $identity = $this->authentication->getIdentity ();
        $customer = null;
        if ($identity)
        {
            $customer = array (
                'id'            => $identity->getId (),
                'first_name'    => $identity->getFirstName (),
                'last_name'     => $identity->getLastName(),
            );
        }

        return array (
            'items'         => array_values ($this->getCartStorage ()->getItems ()),
            'customer'      => $customer,
            'authPage'      => Link::storeLink ('login'),
            'profilePage'   => Link::storeLink ('account', 'orders'),
            'cartPage'      => Link::storeLink ('cart'),
            'comparePage'   => Link::storeLink ('compare'),
            'wishPage'      => Link::storeLink ('wishlist'),

        );
    }

    public function getCompare ()
    {
        return array (
            'items'         => array_values ($this->getCartStorage ()->getCompareItems()),
        );
    }

    public function uploadFile(){
        return false;
        $fileRequest = new \Zend\Http\PhpEnvironment\Request();
        $file = $fileRequest->getFiles()->toArray();
        $globalConfig =  SMStatic::getInstance()->get('store.config');
        $exportdir = $globalConfig['settings']['exportPath'];
        $filename = $exportdir.md5(rand(0,2)). str_replace(' ', '',$file['file']['name']);
        $relativeFilename =  Moto\System::getAbsoluteUrl($filename);
        $absoluteFilename =  Moto\System::getAbsolutePath($filename);
        $validator_zip = new \Zend\Validator\File\IsCompressed();
        $validator_image = new \Zend\Validator\File\IsImage();
        if ($validator_zip->isValid($file['file']['tmp_name']) || $validator_image->isValid($file['file']['tmp_name'])) {
            // file is valid
            move_uploaded_file($file['file']['tmp_name'], $absoluteFilename);
            return array (
                'file'         => $relativeFilename,
            );
        } else {
            return false;
        }

    }


    public function getWishList(){
        /** @var Customer $identity */
        $identity = $this->authentication->getIdentity ();
        $wishlist = array();
        if ($identity)
        {
            $customer = SMStatic::getInstance ()
                ->get ('MotoStore\EntityManager')
                ->getRepository ('MotoStore\Customer\Entity\Customer')
                ->find ($identity->getId ());
            $wishlist_json = $customer->getWishList();
            $wishlist = json_decode($wishlist_json,true);
            return array (
                'items'         => array_values ($wishlist),
            );
        }
        return array (
            'items'         => $wishlist,
        );
    }

    public function removeWishItem($product_id){
        /** @var Customer $identity */
        $identity = $this->authentication->getIdentity ();
        $wishlist = null;
        if ($identity)
        {
            $customer = SMStatic::getInstance ()
                ->get ('MotoStore\EntityManager')
                ->getRepository ('MotoStore\Customer\Entity\Customer')
                ->find ($identity->getId ());

            $wishlist = $customer->getWishList();
            $wishlist_array = json_decode($wishlist,true);
            foreach ( $wishlist_array as $k=>$item){
                if ($item['id'] == $product_id){
                    unset ($wishlist_array [$k]);
                }
            }

            $customer->setWishList(json_encode($wishlist_array));
            $this->entityManager->persist($customer);
            $this->entityManager->flush();
            $this
                ->getEventManager ()
                ->trigger ('store.event.cart.updatewish', null, $wishlist_array);

        }

    }

    public function addWishItem(){


        $params = $this
            ->_request
            ->getParams ();

        $product_id = !empty ($params ['product_id']) ? $params ['product_id'] : 0;
        /** @var Product $product */
        $product = $this->entityManager
            ->getRepository ('MotoStore\Product\Entity\Product')
            ->find ($product_id);
        if ($product)
        {
            /** @var Customer $identity */
            $identity = $this->authentication->getIdentity ();
            if ($identity)
            {
                $customer = SMStatic::getInstance ()
                    ->get ('MotoStore\EntityManager')
                    ->getRepository ('MotoStore\Customer\Entity\Customer')
                    ->find ($identity->getId ());
                $wishlist_json = $customer->getWishList();
                $wishlist_array = array();
                if ($wishlist_json !== null){
                    $wishlist_array = json_decode($wishlist_json,true);
                }
                $id  = $product->getId ();
                if (!isset($wishlist_array[$id])){
                    $wishlist_array [$id] = array(
                        'id'            => $id,
                        'product_id'    => $id,
                    );
                }
                $wishlist_json = json_encode($wishlist_array);
                $customer->setWishList($wishlist_json);
                $this->entityManager->persist($customer);
                $this->entityManager->flush();

                $this
                    ->getEventManager ()
                    ->trigger ('store.event.cart.updatewish', null, $wishlist_array);
            }
        }

    }



    /**
     * Apply Discount to cart items
     *
     * @return bool
     */
    public function applyDiscount ()
    {
        $params = $this
            ->_request
            ->getParams ();

        if (!empty ($params ['discount_code']))
        {
            $query = $this
                ->entityManager
                ->createQueryBuilder ();

            $date  = new \DateTime ();

            $discounts = $query
                ->select (array ('d'))
                ->from ('MotoStore\Settings\Entity\Discount', 'd')
                ->where (
                    $query->expr ()->eq (
                        'lower(d.code)',
                        $query->expr ()->literal (strtolower ($params ['discount_code']))
                    )
                )
                ->andWhere ($query->expr ()->lt ('d.usage', 'd.quantity'))
                ->andWhere ($query->expr ()->gte ('d.date_end', ':now'))
                ->andWhere ($query->expr ()->lte ('d.date_start', ':now'))
                ->setParameter (':now', $date->format ('Y-m-d'))
                ->getQuery()
                ->getArrayResult ();

            if ($discounts)
            {
                /**
                 * @var Discount $first
                 */
                $first = current ($discounts);

                $this->getCartStorage ()->setDiscount ($first);

                return true;
            }
        }
        return false;
    }

    /**
     * @return Storage
     */
    private function getCartStorage ()
    {
        if (!$this->cartStorage)
        {
            $this->cartStorage = new Storage ();
        }
        return $this->cartStorage;
    }

    /**
     * @return EventManager
     */
    private function getEventManager ()
    {
        return SMStatic::getInstance ()->get ('store.eventmanager');
    }

}