<?php


namespace MotoStore\Order\Service;


use MotoStore\Collection\AbstractExport;
use MotoStore\Collection\Collection;
use MotoStore\Collection\ExportInterface;
use MotoStore\Content\Content\DI\SMStatic;
use MotoStore\Order\Entity\Order;

class OrderExport extends AbstractExport implements ExportInterface
{
    /**
     * CSV delimiter
     * @var string
     */
    private $delimiter = ';';

    /**
     * CSV enclosure
     * @var string
     */
    private $enclosure = '"';

    /**
     * Collection to Export
     * @var Collection
     */
    private $collection;

    private $dates;

    /**
     * Target resource
     * @var
     */
    private $resource;

    private $categories = array ();

    /**
     * AbstractExport constructor.
     * @param $target
     * @param Collection $collection
     */
    public function __construct ($target, Collection $collection, $dates = null)
    {
        $this->target = $target;
        $this->collection = $collection;
        $this->dates = $dates;
    }


    /**
     * CSV columns
     * @return array
     */
    public function getHeaders ()
    {
        return array (
            'Customer number',
            'Order number',
            'Date of the order',
            'Salutation invoicing address',
            'Company name invoicing address',
            'Name invoicing address',
            'First name invoicing address',
            'Extension invoicing address',
            'Street invoicing address',
            'Postcode invoicing address',
            'Place invoicing address',
            'Country invoicing address',
            'E-mail orderer',
            'Telephone number',
            'Date of birth orderer',
            'Language for correspondence',
            'Means of correspondence',
            'Claim/receivable type',
            'Article number',
            'Name of article/item',
            'Number of articles/items',
            'Unit price net',
            'Unit price gross',
            'Total price net',
            'Total price gross',
            'Total price after discount net',
            'Total price after discount gross',
            'VAT rate',
            'Claim/receivables group',
            'Salutation delivery address',
            'Company name delivery address',
            'Name delivery address',
            'First name delivery address',
            'Extension delivery address',
            'Street delivery address',
            'Postcode delivery address',
            'Place delivery address',
            'Country delivery address',
            'AccountID',
            'Crefo number',
            'Status',
            'Signal',
            'SolvenyConstraint'

        );

    }

    /**
     * Prepare record before put to csv
     *
     * @param $record
     * @return array
     */
    public function prepareRecord ($record)
    {
        $record['Customer number'] = $record['customer_id'];
        $record['Order number'] = $record['order_id'];
        /** @var \DateTime */
        $createdDate   = $record ['created_at'];
        $record ['Date of the order'] = $createdDate ? $createdDate->format ('d.m.Y') : '';
        $record ['Salutation invoicing address'] = '';
        $record ['Company name invoicing address'] = $record ['company'];
        $record ['Name invoicing address'] = $record['last_name'];
        $record ['First name invoicing address'] = $record['first_name'];
        $record ['Extension invoicing address'] = $record['address2'];
        $record ['Street invoicing address'] = $record['address1'];
        $record ['Postcode invoicing address'] = $record['post_code'];
        $record ['Place invoicing address'] = $record['city'];
        $country  = $record['country'];
        $record ['Country invoicing address'] = $country['iso_3'];
        $record ['E-mail orderer'] = $record['email'];
        $record ['Telephone number'] = $record['contact_phone'];
        $record ['Date of birth orderer'] = '';
        $record ['Language for correspondence'] = '';
        $record ['Means of correspondence'] = '0';
        $record ['Claim/receivable type'] = '0';
        $product = $record['products'][0];
        $record ['Article number'] = $product['id'];
        $record ['Name of article/item'] = $product['name'];
        $record ['Number of articles/items'] = $product['quantity'];
        $record ['Unit price net'] = $product['price'];
        $record ['Unit price gross'] = $product['price'];
        $record ['Total price net'] = $record['total_price'];
        $record ['Total price gross'] = $record['total_price'];
        $record ['Total price after discount net'] = $record['total_price'];
        $record ['Total price after discount gross'] = $record['total_price'];
        $record ['VAT rate'] = isset($record['tax_price'])?$record['tax_price']:0;
        $record ['Claim/receivables group'] = '0';
        $record ['Salutation delivery address'] = '';
        $record ['Company name delivery address'] = $record ['company'];
        $record ['Name delivery address'] = $record['last_name'];
        $record ['First name delivery address'] = $record['first_name'];
        $record ['Extension delivery address']  = $record['address2'];
        $record ['Street delivery address'] = $record['address1'];
        $record ['Postcode delivery address'] = $record['post_code'];
        $record ['Place delivery address'] = $record['city'];
        $record ['Country delivery address'] = $country['iso_3'];
        $record ['AccountID'] = '';
        $record ['Crefo number'] = '';
        $record ['Status'] = '';
        $record ['Signal'] = '';
        $record ['SolvenyConstraint'] = '';

        return $record;
    }



    public function export ()
    {
        $date_from = date("Ymd", strtotime($this->dates['from']));
        $date_to = date("Ymd", strtotime($this->dates['to']));
        $this->open ();
        $records_without_ids = array();
        $records = array();
        $tmp_records = $this->collection
            ->getCollection ();
        foreach ($tmp_records as $record){
            $createdDate   = $record ['created_at'];
            $formatted_date =  $createdDate ? $createdDate->format ('Ymd') : '';
            //error_log(json_encode(array($formatted_date, $date_from, $date_to)));
            if ($formatted_date >= $date_from && $formatted_date <= $date_to){
                $records_without_ids[] = $this->prepareRecord($record);
            }

        }
        foreach ($records_without_ids as $key=>$record){
            $record['id'] = $key;
            $records[] = $record;
        }

        $this->write ($this->getHeaders ());

        foreach ($records as $record)
        {
            $this->write ($this->excludeFields ($record));
        }
        $this->close ();
    }

    /**
     * Filter record by headers
     * @param $record
     * @return array
     */
    private function excludeFields ($record)
    {
        $headers = array_flip ($this->getHeaders ());

        return array_intersect_key (array_merge ($headers, $record), $headers);
    }

    /**
     * Create Target Resource
     */
    private function open ()
    {
        $this->resource = fopen ($this->target, 'w');
    }

    /**
     * Write record to CSV
     * @param array $record
     */
    private function write (array $record)
    {
        fputcsv ($this->resource, $record, $this->delimiter, $this->enclosure);
    }

    /**
     * Close resource
     */
    private function close ()
    {
        return fclose ($this->resource);
    }


}