<?php

/***********************************************************************************************************************
 * @author: <for.samohin@gmail.com>
 **********************************************************************************************************************/
namespace MotoStore\Payment\Authorize;

use MotoStore\Content\Helper\Link;
use MotoStore\Order\Entity\Order;
use MotoStore\Order\Entity\Transaction;
use MotoStore\Payment\Gateway\PaymentModule;
use Zend\Http\PhpEnvironment\Request;
use net\authorize\api\contract\v1 as AnetAPI;
use net\authorize\api\controller as AnetController;


class Authorize extends PaymentModule
{

    protected  $_apiUrl = 'https://accept.authorize.net/payment/payment';
    protected  $_apiUrlSandbox = 'https://test.authorize.net/payment/payment';

    public function getName()
    {
        return 'Authorize';
    }

    public function getImage()
    {
        return 'authorize.png';
    }

    public function getDescription()
    {
        return '';
    }

    public function getHelpText()
    {
        return 'Login ID , Transaction Key and Signature Key you can get in the Authorize.NET Merchant Interface.';
    }

    public function setVariables()
    {
        $this
            ->addVariable (
                self::OPTION_DISPLAY_NAME,
                array(
                    'name' => 'Display Name',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => 'Authorize.NET',
                    'required' => true,
                )
            )->addVariable (
                'login_id',
                array(
                    'name' => 'API Login ID',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )->addVariable (
                'transaction_key',
                array(
                    'name' => 'API Transaction Key',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )->addVariable (
                'hash_key',
                array(
                    'name' => 'Signature Key',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )->addVariable (
                'test_mode',
                array(
                    'name' => 'Sandbox mode',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => 'The context mode of this module',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            )->addVariable (
                self::OPTION_ENABLED,
                array(
                    'name' => 'Enabled',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            );

    }

    /**
     * @param Order $order
     * @return mixed|void
     */
    public function checkout(Order $order)
    {
        $currency = $this->getCurrency ();

        $merchantAuthentication = new AnetAPI\MerchantAuthenticationType();
        $merchantAuthentication->setName($this->getOption ('login_id'));
        $merchantAuthentication->setTransactionKey($this->getOption ('transaction_key'));
        $refId = $order->getOrderId ();

        $transactionRequestType = new AnetAPI\TransactionRequestType();
        $transactionRequestType->setTransactionType("authCaptureTransaction");
        $transactionRequestType->setAmount(number_format ($order->getTotalPrice () * $currency->getExchangeRate (), 2, '.', ''));


        $setting1 = new AnetAPI\SettingType();
        $setting1->setSettingName("hostedPaymentButtonOptions");
        $setting1->setSettingValue("{\"text\": \"Pay\"}");

        $setting2 = new AnetAPI\SettingType();
        $setting2->setSettingName("hostedPaymentOrderOptions");
        $setting2->setSettingValue("{\"show\": false}");

        $setting3 = new AnetAPI\SettingType();
        $setting3->setSettingName("hostedPaymentReturnOptions");

        $return_options = [
            'url'=> Link::storeLink('checkout', 'success', array('provider' => $this->getName()), false),
            'cancelUrl'=> Link::storeLink('checkout', 'success', array('provider' => $this->getName()), false),
            'showReceipt'=> true
        ];
        $setting3->setSettingValue(json_encode($return_options));

        $request = new AnetAPI\GetHostedPaymentPageRequest();
        $request->setMerchantAuthentication($merchantAuthentication);
        $request->setRefId($refId);
        $request->setTransactionRequest($transactionRequestType);

        $request->addToHostedPaymentSettings($setting1);
        $request->addToHostedPaymentSettings($setting2);
        $request->addToHostedPaymentSettings($setting3);


        $controller = new AnetController\GetHostedPaymentPageController($request);

        if ($this->getOption ('test_mode') == 0){
            $response = $controller->executeWithApiResponse(\net\authorize\api\constants\ANetEnvironment::PRODUCTION);
        } else{
            $response = $controller->executeWithApiResponse(\net\authorize\api\constants\ANetEnvironment::SANDBOX);
        }

        $token = '';

        if (($response != null) && ($response->getMessages()->getResultCode() == "Ok")) {
            $token =  $response->getToken();
        } else {

            // Order Failed
           // error_log(json_encode($response->getMessages()->getMessage()));
            Link::redirect ('checkout', 'success');
        }

        $url= ($this->getOption ('test_mode') == 0)? $this->_apiUrl : $this->_apiUrlSandbox;
        $data = array(
            'token'                  => $token
        );

        echo $this->form($url, $data);

    }

    public function handle(Request $request)
    {
        $order = $this->getCurrentOrder();
        if ($request->getPost('x_SHA2_Hash')){

            $hashFields =  [
                $_POST['x_trans_id'],
                $_POST['x_test_request'],
                $_POST['x_response_code'],
                $_POST['x_auth_code'],
                $_POST['x_cvv2_resp_code'],
                $_POST['x_cavv_response'],
                $_POST['x_avs_code'],
                $_POST['x_method'],
                $_POST['x_account_number'],
                $_POST['x_amount'],
                $_POST['x_company'],
                $_POST['x_first_name'],
                $_POST['x_last_name'],
                $_POST['x_address'],
                $_POST['x_city'],
                $_POST['x_state'],
                $_POST['x_zip'],
                $_POST['x_country'],
                $_POST['x_phone'],
                $_POST['x_fax'],
                $_POST['x_email'],
                $_POST['x_ship_to_company'],
                $_POST['x_ship_to_first_name'],
                $_POST['x_ship_to_last_name'],
                $_POST['x_ship_to_address'],
                $_POST['x_ship_to_city'],
                $_POST['x_ship_to_state'],
                $_POST['x_ship_to_zip'],
                $_POST['x_ship_to_country'],
                $_POST['x_invoice_num'],
            ];
            $hashString = '^'.implode('^', $hashFields).'^';

            $generatedhash = strtoupper(HASH_HMAC('sha512', $hashString, hex2bin($this->getOption ('hash_key'))));

            $order = $this->getCurrentOrder ($request->getPost ('x_invoice_num'));

            if ($order && $request->getPost('x_response_code') != '' &&  ($request->getPost('x_SHA2_Hash')
                    ==
                    $generatedhash)
            )
            {
                $em = $this->serviceLocator->get ('MotoStore\EntityManager');
                $amount           = $request->getPost('x_amount');
                $hash             = $request->getPost('x_SHA2_Hash');


                if ( $request->getPost ('x_response_code') == 1 ){

                    $status = Transaction::PAYMENT_STATUS_PAID;
                    $orderStatus = Order::ORDER_PAYMENT_ACCEPTED;
                    $this->getCart ()->setIsSuccess (true);
                    \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_PAID",$order->getOrderId());

                }
                else{
                    error_log ('Authorized Err: ' . json_encode($request->getPost()));

                    $status = Transaction::PAYMENT_STATUS_DECLINED;
                    $orderStatus = Order::ORDER_STATUS_CANCELED;
                }

                $transaction = new Transaction ();
                $transaction->setOrder ($order);
                $transaction->setTransactionId ($request->getPost('x_trans_id'));

                $date = new \DateTime ();

                $transaction->setDate ($date);

                $transaction->setDate ($date);
                $transaction->setPaymentMethod($this->getName ());
                $transaction->setStatus ($status);
                $transaction->setAdditional(json_encode ($_REQUEST));

                $em->persist ($transaction);

                $order->addTransaction ($transaction);
                $order->setStatus ($orderStatus);

                $em->merge ($order);
                $em->flush ();
            }
        }
        $this->getCart ()->setIsSuccess (true);
    }

    public function generateSH512($textToHash, $signatureKey)
    {
        if ($textToHash != null && $signatureKey != null) {
            $sig = hash_hmac('sha512', $textToHash, hex2bin($signatureKey));
            return strtoupper($sig);
        }
        return '';
    }



}