<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com>
 **********************************************************************************************************************/

namespace MotoStore\Payment\Payone;


use MotoStore\Content\Helper\Link;
use MotoStore\Content\Helper\Price;
use MotoStore\Content\Presenter\CheckoutPresenter;
use MotoStore\Order\Entity\Order;
use MotoStore\Order\Entity\Product;
use MotoStore\Order\Entity\Transaction;
use MotoStore\Payment\Gateway\PaymentModule;
use Zend\Http\Client;
use Zend\Http\PhpEnvironment\Request;

class Payone extends PaymentModule
{
    const PAYONE_SERVER_API_URL = 'https://secure.pay1.de/client-api/';
    private static $cardTypes = array (
        'V' => 'Visa',
        'M' => 'MasterCard',
        'A' => 'American Express',
        'D' => 'Diners',
        'J' => 'JCB',
        'O' => 'Maestro International',
        'C' => 'Discover',
        'B' => 'Carte Bleue',
    );

    /**
     * Get payment processor
     *
     * @return string
     */
    public function getName ()
    {
        return 'Payone';
    }

    /**
     * Get Payment Icon
     * @return string
     */
    public function getImage ()
    {
        return 'payone.jpeg';
    }

    /**
     * description
     * @return string
     */
    public function getDescription ()
    {
        return '';
    }

    /**
     * Help text
     * @return string
     */
    public function getHelpText()
    {
        return "<p>Please, disable \"3-D Secure\" risckcheck in your <b>Payone</b> account</p>";
    }


    protected $hashParameters = array(
        'mid',
        'amount',
        'productid',
        'aid',
        'currency',
        'accessname',
        'portalid',
        'due_time',
        'accesscode',
        'mode',
        'storecarddata',
        'access_expiretime',
        'request',
        'checktype',
        'access_canceltime',
        'responsetype',
        'addresschecktype',
        'access_starttime',
        'reference',
        'consumerscoretype',
        'access_period',
        'userid',
        'invoiceid',
        'access_aboperiod',
        'customerid',
        'invoiceappendix',
        'access_price',
        'param',
        'invoice_deliverymode',
        'access_aboprice',
        'narrative_text',
        'eci',
        'access_vat',
        'successurl',
        'settleperiod',
        'errorurl',
        'settletime',
        'backurl',
        'vaccountname',
        'exiturl',
        'vreference',
        'clearingtype',
        'encoding',
    );

    /**
     * Set Payment Variables Array
     * @return array
     */
    public function setVariables()
    {
        $this
            ->addVariable (
                self::OPTION_DISPLAY_NAME,
                array(
                    'name' => 'Display Name',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => 'Payone',
                    'required' => true,
                )
            )
            ->addVariable (
                'merchant_id',
                array(
                    'name' => 'Merchant ID',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'portal_id',
                array(
                    'name' => 'Portal ID',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'portal_key',
                array(
                    'name' => 'Portal Key',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'account_id',
                array(
                    'name' => 'Account ID',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'testmode',
                array(
                    'name' => 'Test Mode',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            )
            ->addVariable (
                'language',
                array(
                    'name' => 'Language',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 'EN',
                    'required' => true,
                    'options' => array (
                        'EN' => 'EN',
                        'DE' => 'DE',
                        'ES' => 'ES',
                        'FR' => 'FR',
                        'IT' => 'IT',
                        'PL' => 'PL',
                        'GR' => 'GR',
                        'RO' => 'RO',
                        'RU' => 'RU',
                        'TR' => 'TR',
                        'CN' => 'CN',
                        'CZ' => 'CZ',
                        'NL' => 'NL',
                        'DA' => 'DA',
                        'SV' => 'SV'
                    ),
                )
            )
            ->addVariable (
                self::OPTION_ENABLED,
                array(
                    'name' => 'Enabled',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            )->addVariable (
                'numberLabel',
                array(
                    'name' => 'Card number label',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => 'Card Number',
                    'required' => true,
                )
            )->addVariable (
                'expireLabel',
                array(
                    'name' => 'Card expiry date label',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => 'Expiry date',
                    'required' => true,
                )
            )->addVariable (
                'cvcLabel',
                array(
                    'name' => 'Credit verification number (CVC) label',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => 'CVC',
                    'required' => true,
                )
            )->addVariable (
                'submitLabel',
                array(
                    'name' => 'Submit button label',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => 'Submit Payment',
                    'required' => true,
                )
            );
    }

    /**
     * Proceed payment
     *
     * @param Order $order
     * @return mixed
     */
    public function checkout (Order $order)
    {
        $data = array (
            'request' => 'authorization',
            'clearingtype' => 'cc',
            'reference' => $order->getOrderId (),
            'aid'       => $this->getOption ('account_id'),
            'mid'       => $this->getOption ('merchant_id'),
            'portalid'  => $this->getOption ('portal_id'),
            'mode'      => $this->getOption ('testmode') ? 'test' : 'live',
            'encoding'  => 'UTF-8',
            'amount' => $order->getTotalPrice () * 100,
            'currency' => $this->getCurrency()->getCode (),
            'successurl' => (Link::storeLink ('checkout', 'success', array ('provider' => $this->getName (), 'reference' => $order->getOrderId ()), false)),
            'errorurl' => (Link::storeLink ('cart' , null, array(), false)),
            'backurl' => Link::storeLink ('checkout', 'success', array ('provider' => $this->getName (), 'reference' => $order->getOrderId ()), false)
        );

        $data['hash'] = $this->createRequestHash ($data);


        $request = new Request ();

        $data ['payoneCards'] = self::$cardTypes;
        $data ['cardpan'] = '';
        $data ['cardtype'] = 'V';
        $data ['expiry_card_date'] = '';
        $data ['cardcvc2'] = '';


        $data ['payment_method'] = $this->getName ();
        $data ['payment_amount'] = Price::decorate($order->getTotalPrice(), $this->getCurrency ());
        $data ['country'] = ($order->getCountry ())? $order->getCountry ()->getIso2 ():'';
        $data ['language'] = strtolower($this->getOption ('language', 'en'));


        $data = array_merge ($data, $this->getLocalizationOptions());


        if ($request->isPost ())
        {
            $data ['cardpan'] = preg_replace('[\D]', '', $request->getPost ('number'));
            $data ['cardtype'] = $request->getPost ('cardtype');
            $data ['expiry_card_date'] = $request->getPost ('expiry', '');

            preg_match ("/([0-9]{2})(.+)?([0-9]{2})/", $data ['expiry_card_date'], $expireData);

            $data ['cardexpiredate'] = '';

            if (isset($expireData [3]) && isset($expireData [1]))
            {
                $data ['cardexpiredate'] =  $expireData [3] . $expireData [1];
            }

            $data ['cardcvc2'] = $request->getPost ('cvc');
            $data ['payment_method'] = $this->getName ();
            $data ['lastname'] = $order->getLastName();

            $data ['customermessage'] = '';

            $result = $this->send ($data);
            //$this->debug ($result);

            if (isset($result->customermessage))
            {
                $data ['customermessage'] = $result->customermessage;
            }
            else
            {
                if (isset($result->status) && $result->status === 'APPROVED')
                {
                    $this->completeOrder ($order, $result);
                    Link::redirect ('checkout', CheckoutPresenter::STEP_SUCCESS, array ('provider' => $this->getName (), 'reference' => $order->getOrderId ()));
                }
                if (isset($result->status) && $result->status === 'REDIRECT')
                {
                    header ('Location: ' . $result->redirecturl);
                    exit;
                }
            }
        }

        echo $this
            ->serviceLocator
            ->get ('store.twig')
            ->render ('store/payment/card.twig', $data);
        exit;

    }

    /**
     * Handle Success
     *
     * @return mixed
     */
    public function handle (Request $request)
    {

        $order = $this->getCurrentOrder ($request->getPost('reference', 0));
        $em = $this->serviceLocator->get ('MotoStore\EntityManager');

        if ($order)
        {
            $invoice_id     = $request->getQuery ('mb_transaction_id');
            $status         = (int) $request->getQuery ('status', -2);

            $transaction = new Transaction ();
            $transaction->setOrder ($order);
            $transaction->setTransactionId ($invoice_id);


            $date = new \DateTime ();


            $transaction->setDate ($date);
            $transaction->setPaymentMethod($this->getName ());

            switch ($status)
            {
                case 2:
                    $transactionStatus = Transaction::PAYMENT_STATUS_PAID;
                    $orderStatus = Order::ORDER_PAYMENT_ACCEPTED;
                    \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_PAID",$order->getOrderId());
                    break;
                case 0:
                    $transactionStatus = Transaction::PAYMENT_STATUS_PENDING;
                    $orderStatus = Order::ORDER_STATUS_AWAITING_FULFILLMENT;
                    break;
                case -1:
                case -2:
                    $transactionStatus = Transaction::PAYMENT_STATUS_DECLINED;
                    $orderStatus = Order::ORDER_STATUS_CANCELED;
                    break;
                default:
                    $transactionStatus = Transaction::PAYMENT_STATUS_DECLINED;
                    $orderStatus = Order::ORDER_STATUS_CANCELED;
            }

            $transaction->setStatus ($transactionStatus);
            $order->setStatus ($orderStatus);

            $transaction->setAdditional(json_encode ($request->getQuery ()));

            $em->persist ($transaction);

            $order->addTransaction ($transaction);

            $em->merge ($order);
            $em->flush ();
            $this->getCart ()->setIsSuccess (true);
        }
    }

    private function completeOrder (Order $order, $result)
    {
        $em = $this->serviceLocator->get ('MotoStore\EntityManager');

        $transaction = new Transaction ();
        $transaction->setOrder ($order);
        $transaction->setTransactionId ($result->txid);

        $transaction->setStatus (Transaction::PAYMENT_STATUS_PAID);
        $order->setStatus (Order::ORDER_PAYMENT_ACCEPTED);

        $date = new \DateTime ();

        $transaction->setDate ($date);
        $transaction->setPaymentMethod($this->getName ());
        $transaction->setAdditional(json_encode ($result));

        $em->persist ($transaction);

        $order->addTransaction ($transaction);

        $em->merge ($order);
        $em->flush ();

        $this->getCart ()->setIsSuccess (true);
    }

    private function createRequestHash ($data)
    {
        ksort($data);
        $signOut = implode ('', $data);
        return hash ('md5', $signOut . $this->getOption ('portal_key'));
    }


    protected function send ($data = array ())
    {
        $result = array ();

        $this->debug ($data);
        $client = new Client (self::PAYONE_SERVER_API_URL);
        $client
            ->setOptions (array (
                'maxredirects'  => 0,
                'timeout'       => 60,
                'sslverifypeer' => false,
            ))
            ->setMethod (Request::METHOD_GET)
            ->setParameterGet ($data);


        $response = $client->send ();

        if ($response->isSuccess ())
        {
            return $this->parseResponse($response->getBody ());
        }
        return $result;
    }

    /**
     * @param $response
     * @return array
     */
    public static function parseResponse($response)
    {
        return json_decode($response);
    }

    private function getLocalizationOptions ()
    {
        return array (
            'form_numberLabel' => $this->getOption('numberLabel', ''),
            'form_expireLabel' => $this->getOption('expireLabel', ''),
            'form_cvcLabel' => $this->getOption('cvcLabel', ''),
            'form_submitLabel' => $this->getOption('submitLabel', ''),
        );
    }
}