<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Payment\PostFinance;


use MotoStore\Content\Helper\Link;
use MotoStore\Order\Entity\Order;
use MotoStore\Order\Entity\Product;
use MotoStore\Order\Entity\Transaction;
use MotoStore\Payment\Gateway\PaymentModule;
use Zend\Http\PhpEnvironment\Request;

class PostFinance extends PaymentModule
{
    /**
     * Get payment processor
     *
     * @return string
     */
    public function getName ()
    {
        return 'PostFinance';
    }

    /**
     * Get Payment Icon
     * @return string
     */
    public function getImage ()
    {
        return 'PostFinance.jpg';
    }

    /**
     * description
     * @return string
     */
    public function getDescription ()
    {
        return '';
    }

    /**
     * Help text
     * @return string
     */
    public function getHelpText()
    {
        return "";
    }

    /**
     * Set Payment Variables Array
     * @return array
     */
    public function setVariables()
    {
        $this
            ->addVariable (
                self::OPTION_DISPLAY_NAME,
                array(
                    'name' => 'Display Name',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => 'PostFinance',
                    'required' => true,
                )
            )
            ->addVariable (
                'pspid',
                array(
                    'name' => 'STORE.PAYMENT.POSTFINANCE.PSPID',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'hash',
                array(
                    'name' => 'Hash algorithm',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 'sha256',
                    'required' => true,
                    'options' => array (
                        'sha1' => 'SHA-1',
                        'sha256' => 'SHA-256',
                        'sha512' => 'SHA-512',
                    ),
                )
            )
            ->addVariable (
                'language',
                array(
                    'name' => 'Language',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 'en_US',
                    'required' => true,
                    'options' => array (
                        'en_US' => 'EN',
                        'de_DE' => 'DE',
                        'es_ES' => 'ES',
                        'fr_FR' => 'FR',
                        'it_IT' => 'IT',
                    ),
                )
            )
            ->addVariable (
                'sha',
                array(
                    'name' => 'SHA-IN',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'sha_out',
                array(
                    'name' => 'SHA-OUT',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'testmode',
                array(
                    'name' => 'Test Mode',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            )
            ->addVariable (
                self::OPTION_ENABLED,
                array(
                    'name' => 'Enabled',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            );
    }

    /**
     * Handle Success
     *
     * @return mixed
     */
    public function handle (Request $request)
    {

        $order      = $this->getCurrentOrder ();
        $em         = $this->serviceLocator->get ('MotoStore\EntityManager');
        $order_id   = $request->getQuery ('orderID');

        if (empty($order) && !empty ($order_id))
        {
            $order = $this->getCurrentOrder ($order_id);
        }

        if ($order && $this->checkSignOut($request->getQuery()->toArray ()))
        {

            $invoice_id     = $request->getQuery ('PAYID') ? $request->getQuery ('PAYID') : 'noid';
            $status         = $request->getQuery ('STATUS');
            $ncerror        = $request->getQuery ('NCERROR');

            $transaction = new Transaction ();
            $transaction->setOrder ($order);
            $transaction->setTransactionId ($invoice_id);

            $date = new \DateTime ();

            $transaction->setDate ($date);
            $transaction->setPaymentMethod($this->getName ());

            if (0 == $ncerror)
            {

                switch ($status)
                {
                    // Accepted
                    case 5:
                    case 9:
                        $transactionStatus  = Transaction::PAYMENT_STATUS_PAID;
                        $orderStatus        = Order::ORDER_PAYMENT_ACCEPTED;
                        \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_PAID",$order->getOrderId());
                        break;

                    // Declined
                    case 2:
                    case 93:
                    case 1:
                        $transactionStatus  = Transaction::PAYMENT_STATUS_DECLINED;
                        $orderStatus        = Order::ORDER_STATUS_CANCELED;
                        break;

                    // Pending
                    case 41:
                    case 51:
                    case 91:
                    case 4:
                        $transactionStatus  = Transaction::PAYMENT_STATUS_PENDING;
                        $orderStatus        = Order::ORDER_STATUS_AWAITING_FULFILLMENT;
                        break;

                    default:
                        $transactionStatus  = Transaction::PAYMENT_STATUS_PENDING;
                        $orderStatus        = Order::ORDER_STATUS_CANCELED;
                }

                $transaction->setStatus ($transactionStatus);
                $order->setStatus ($orderStatus);
            }
            else
            {
                $transaction->setStatus (Transaction::PAYMENT_STATUS_DECLINED);
            }

            $transaction->setAdditional(json_encode ($request->getQuery ()));

            $em->persist ($transaction);

            $order->addTransaction ($transaction);

            $em->merge ($order);
            $em->flush ();

            $this->getCart ()->setIsSuccess (true);
        }

    }

    private function checkSignOut ($params = array())
    {
        $signin = isset ($params ['SHASIGN']) ? $params ['SHASIGN'] : '';

        unset($params ['SHASIGN']);
        unset($params ['url']);
        unset($params ['provider']);

        return $this->getDigest($params, $this->getOption ('sha_out')) === $signin;

    }

    private function isExcluded ($value) {
        return $value === '';
    }

    /**
     * Proceed payment
     *
     * @param Order $order
     * @return mixed
     */
    public function checkout(Order $order)
    {
        $currency = $this->getCurrency ();

        $names = array_map (function (Product $p) {
            return $p->getName();
        }, $order->getProducts ()->toArray ());


        $paymentData = array (
            'PSPID'        => $this->getOption ('pspid'),
            'ORDERID'      => $order->getOrderId (),
            'AMOUNT'       => number_format ($order->getTotalPrice (), 2, '.', '') * 100,
            'CURRENCY'     => $currency->getCode (),
            'LANGUAGE'     => $this->getOption ('language', 'en_US'),
            'CN'           => $order->getFirstName () . ' ' . $order->getLastName (),
            'EMAIL'        => $order->getEmail (),
            'TITLE'        => 'Order #' . $order->getOrderId () . ' [' .  join (', ', $names) . ']'
        );

        // payment success
        $paymentData ['ACCEPTURL']  = Link::storeLink ('checkout', 'success', array ('provider' => $this->getName ()), false);
        // payment failed
        $paymentData ['DECLINEURL'] = Link::storeLink ('checkout', 'success', array ('provider' => $this->getName ()), false);
        // cancel payment
        $paymentData ['CANCELURL']  = Link::storeLink ('cart', null, array(), false);
        // return from payment
        $paymentData ['BACKURL']    = Link::storeLink ('cart', null, array(), false);

        $paymentData ['SHASIGN'] = $this->getDigest ($paymentData, $this->getOption ('sha'));

        $target = 'https://e-payment.postfinance.ch/ncol/prod/orderstandard.asp';

        if ($this->getOption ('testmode'))
        {
            $target = 'https://e-payment.postfinance.ch/ncol/test/orderstandard.asp';
        }

        echo $this
            ->form ($target, $paymentData);
    }

    /**
     * Get sha params digest
     *
     * @param $params
     * @return string
     */
    public function getDigest($params, $hash)
    {
        $params = array_change_key_case($params, CASE_UPPER);

        uksort($params, 'strnatcasecmp');
        $string = '';

        foreach ($params as $formKey => $formVal)
        {

            if (!$this->isExcluded($formVal))
            {
                $string .= sprintf('%s=%s%s', $formKey, $formVal, $hash);
            }
        }

        return $this->generateDigest ($string);
    }

    /**
     * Create SHA digest
     *
     * @param $hashString
     * @return string
     */
    protected function generateDigest ($hashString)
    {
        return strtoupper (hash($this->getOption ('hash'), $hashString));
    }
}