<?php

namespace MotoStore\Payment\Stripe;


use Moto\Exception;
use MotoStore\Content\Helper\Link;
use MotoStore\Content\Helper\Price;
use MotoStore\Content\Presenter\CheckoutPresenter;
use MotoStore\Order\Entity\Order;
use MotoStore\Order\Entity\Product;
use MotoStore\Order\Entity\Transaction;
use MotoStore\Payment\Gateway\PaymentModule;
use Zend\Http\Client;
use Zend\Http\PhpEnvironment\Request;

class Stripe extends PaymentModule
{
    const STRIPE_API_URL = 'https://api.stripe.com/v1/charges';
    /**
     * Get payment processor
     *
     * @return string
     */
    public function getName ()
    {
        return 'Stripe';
    }

    /**
     * Get Payment Icon
     * @return string
     */
    public function getImage ()
    {
        return 'stripe.png';
    }

    /**
     * description
     * @return string
     */
    public function getDescription ()
    {
        return '';
    }

    /**
     * Help text
     * @return string
     */
    public function getHelpText()
    {
        return "Please, get API keys in your stripe account ( https://dashboard.stripe.com/account/apikeys )<br>
                Also you need to add webhook to your account (https://dashboard.stripe.com/account/webhooks) 
                with event type 'checkout.session.completed' for correct payment processing . 
                Url to use in webhook : ". Link::storeLink ('checkout', 'success', array ('provider' => $this->getName (),'wh'=>true), false);
    }

    /**
     * Set Payment Variables Array
     * @return array
     */
    public function setVariables()
    {
        $this
            ->addVariable (
                self::OPTION_DISPLAY_NAME,
                array(
                    'name' => 'Display Name',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => 'Stripe',
                    'required' => true,
                )
            )
            ->addVariable (
                'key',
                array(
                    'name' => 'Publisable Key',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'secret_key',
                array(
                    'name' => 'Secret Key',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'signing_secret',
                array(
                    'name' => 'Webhook Signing Secret',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                self::OPTION_ENABLED,
                array(
                    'name' => 'Enabled',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            );
    }

    /**
     * Proceed payment
     * @param Order $order
     * @return mixed
     */
    public function checkout (Order $order)
    {

        \Stripe\Stripe::setApiKey($this->getOption ('secret_key'));

        $names = array ();

        foreach ($order->getProducts () as $product)
        {
            $names [] = $product->getName ();
        }
        $session = \Stripe\Checkout\Session::create([
            'payment_method_types' => ['card'],
            'line_items' => [[
                'name' => 'Order #' . $order->getOrderId () ,
                'description' => ' [' .  join (', ', $names) . ']',
                'amount' => ceil ( $order->getTotalPrice () * 100),
                'currency' => $this->getCurrency()->getCode (),
                'quantity' => 1,
            ]],
            'client_reference_id'=> $order->getOrderId (),
            'success_url' => Link::storeLink ('checkout', 'success', array ('provider' => $this->getName (), 'orderid' => $order->getOrderId ()), false),
            'cancel_url' => Link::storeLink ('cart', '', array (), false),
        ]);


        $data = array (
            'session_id' => $session['id'],
            'key' => $this->getOption ('key'),
        );


        echo $this
            ->serviceLocator
            ->get ('store.twig')
            ->render ('store/payment/stripe/paymentform.twig', $data);
        exit;
    }

    /**
     * Handle Success
     *
     * @return mixed
     */
    public function handle (Request $request)
    {
  if (isset($_REQUEST['wh']))
        {
            \Stripe\Stripe::setApiKey($this->getOption ('secret_key'));
            $endpoint_secret = $this->getOption ('signing_secret');

            $payload = @file_get_contents('php://input');
            $sig_header = $_SERVER['HTTP_STRIPE_SIGNATURE'];
            $event = null;

            try {
                $event = \Stripe\Webhook::constructEvent(
                    $payload, $sig_header, $endpoint_secret
                );
            } catch(\UnexpectedValueException $e) {
                // Invalid payload
                http_response_code(400); // PHP 5.4 or greater
                exit();
            } catch(\Stripe\Error\SignatureVerification $e) {
                // Invalid signature
                http_response_code(400); // PHP 5.4 or greater
                exit();
            }

            if ($event->type == 'checkout.session.completed') {
                $session = $event->data->object;
                $order_id = $session->client_reference_id;
                //error_log($order_id);
                $order = $this->getCurrentOrder ($order_id);
                $em = $this->serviceLocator->get ('MotoStore\EntityManager');
                if ($order ) {
                    $transaction = new Transaction ();
                    $transaction->setOrder ($order);
                    $transaction->setTransactionId ($session->payment_intent);
                    $date = new \DateTime ();
                    $transaction->setDate ($date);
                    $transaction->setPaymentMethod($this->getName ());
                    $transactionStatus = Transaction::PAYMENT_STATUS_PAID;
                    $orderStatus = Order::ORDER_PAYMENT_ACCEPTED;
                    \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_PAID",$order->getOrderId());
                    $transaction->setStatus ($transactionStatus);
                    $order->setStatus ($orderStatus);
                    $transaction->setAdditional(json_encode ($session));
                    $em->persist ($transaction);
                    $order->addTransaction ($transaction);
                    $em->merge ($order);
                    $em->flush ();
            }



            http_response_code(200);
                die();
        }



        }
        $this->getCart ()->setIsSuccess (true);
    }

}