<?php

namespace MotoStore\Payment\YandexMoney;


use MotoStore\Content\Helper\Link;
use MotoStore\Order\Entity\Order;
use MotoStore\Order\Entity\Transaction;
use MotoStore\Payment\Gateway\PaymentModule;
use Zend\Http\Client;
use Zend\Http\PhpEnvironment\Request;
use Zend\Uri\UriFactory;
use YandexCheckout\Client as YandexClient;
use YandexCheckout\Common\Exceptions\ApiException;
use YandexCheckout\Request\Payments\Payment\CreateCaptureRequest;
use YandexCheckout\Request\Payments\CreatePaymentRequest;
use YandexCheckout\Request\Payments\CreatePaymentRequestSerializer;
use YandexCheckout\Model\Notification\NotificationSucceeded;
use YandexCheckout\Model\Notification\NotificationWaitingForCapture;
use YandexCheckout\Model\NotificationEventType;
use YandexCheckout\Model\PaymentMethodType;
use YandexCheckout\Model\PaymentStatus;

class YandexMoney extends PaymentModule
{
    /**
     * @var YandexClient
     */
    private $apiClient;

    private $order;

    /**
     * Get payment processor
     *
     * @return string
     */
    public function getName()
    {
        return 'YandexMoney';
    }

    /**
     * Get Payment Icon
     * @return string
     */
    public function getImage ()
    {
        return 'ukassa_logo.jpg';
    }

    /**
     * description
     * @return string
     */
    public function getDescription ()
    {
        return '';
    }

    /**
     * Help text
     * @return string
     */
    public function getHelpText()
    {
        return "<h3>Параметры из личного кабинета Яндекс.Кассы</h3>
       <ul>
        <li>имя пользователя — идентификатор вашего магазина в Яндекс.Кассе;</li>
        <li>пароль — ваш секретный ключ.</li>
        </ul>
        <p>Выпустить секретный ключ (а также перевыпустить и удалить неактуальный) можно в личном кабинете Яндекс.Кассы, в разделе <a href=\"https://kassa.yandex.ru/my/tunes\">Настройки</a>.</p>
        Также вам необходимо установить в личном кабинете Яндекс Кассы:<br/>
        1. Динамичесие    shopSuccessUrl, shopFailUrl <br/>
        2. Ссылку вида 'http(s)://ВАШ.ДОМЕН/store/checkout/success?provider=YandexMoney' в поле Notification URL<br/>
        </p>";
    }

    /**
     * Set Payment Variables Array
     * @return array
     */
    public function setVariables ()
    {
        $this
            ->addVariable (
                self::OPTION_DISPLAY_NAME,
                array(
                    'name' => 'Display Name',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => 'YandexMoney',
                    'required' => true,
                )
            )
            ->addVariable (
                'ShopID',
                array(
                    'name' => 'Shop ID',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )->addVariable (
                'shopPassword',
                array(
                    'name' => 'Secret Key',
                    'type' => self::VARIABLE_TYPE_TEXT,
                    'help' => '',
                    'default' => '',
                    'required' => true,
                )
            )
            ->addVariable (
                'testmode',
                array(
                    'name' => 'Test Mode',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            )
            ->addVariable (
                self::OPTION_ENABLED,
                array(
                    'name' => 'Enabled',
                    'type' => self::VARIABLE_TYPE_SELECT,
                    'help' => '',
                    'default' => 0,
                    'required' => true,
                    'options' => array (
                        0 => 'No',
                        1 => 'Yes'
                    ),
                )
            );
    }

    /**
     * Handle Success
     *
     * @return mixed
     */
    public function handle (Request $request)
    {
        if (
            $_SERVER['REQUEST_METHOD'] == "POST" &&
            isset($_REQUEST['provider'])
            && $_REQUEST['provider'] == 'YandexMoney'
        ) {
            $body = @file_get_contents('php://input');
            $callbackParams = json_decode($body, true);
            if (!json_last_error()) {
                try {
                    $this->processNotification($callbackParams);
                } catch (Exception $e) {
                    error_log("Error while process notification: " . $e->getMessage());
                }
            } else {
                header("HTTP/1.1 400 Bad Request");
                header("Status: 400 Bad Request");
            }
            exit();
        }

    }

    protected function processNotification($callbackParams)
    {
        try {
            $notificationModel = ($callbackParams['event'] === NotificationEventType::PAYMENT_SUCCEEDED)
                ? new NotificationSucceeded($callbackParams)
                : new NotificationWaitingForCapture($callbackParams);

        } catch (\Exception $e) {
            error_log('Invalid notification object - '.$e->getMessage());
            header("HTTP/1.1 400 Bad Request");
            header("Status: 400 Bad Request");
            exit();
        }


        $payment = $notificationModel->getObject();

        $order = $this->getCurrentOrder ();
        if (isset($callbackParams['object']['metadata']) && isset($callbackParams['object']['metadata']['order_id'])){
            $order = $this->getCurrentOrder ($callbackParams['object']['metadata']['order_id']);
        }
        $this->order = $order;
        $em = $this->serviceLocator->get ('MotoStore\EntityManager');

        if (!$order) {
            header("HTTP/1.1 404 Not Found");
            header("Status: 404 Not Found");
            exit();
        }

        if ($payment->getStatus() === PaymentStatus::SUCCEEDED) {
            $transaction = new Transaction ();
            $transaction->setOrder ($order);
            $transaction->setTransactionId ($payment->getId());
            $date = new \DateTime ();
            $transaction->setDate ($date);
            $transaction->setPaymentMethod($this->getName ());
            $transactionStatus  = Transaction::PAYMENT_STATUS_PAID;
            $orderStatus        = Order::ORDER_PAYMENT_ACCEPTED;
            $transaction->setStatus ($transactionStatus);
            $order->setStatus ($orderStatus);
            $transaction->setAdditional(json_encode ($callbackParams));
            $em->persist ($transaction);
            $order->addTransaction ($transaction);
            $em->merge ($order);
            $em->flush ();
            $this->getCart ()->setIsSuccess (true);
            \Moto\Hook::trigger("MOTOSTORE_HOOK_ORDER_PAID",$order->getOrderId());
        } elseif ($payment->getStatus() === PaymentStatus::WAITING_FOR_CAPTURE) {
            $this->capturePayment($this->getApiClient(), $payment->getId());
        } else {
            // error_log('Wrong payment status: '.$payment->getStatus());
            header("HTTP/1.1 402 Payment Required");
            header("Status: 402 Payment Required");
        }

        exit();



    }
    /**
     * Proceed payment
     *
     * @param Order $order
     * @return mixed
     */
    public function checkout (Order $order)
    {
        $this->order = $order;
        $builder        = $this->getBuilder($order);
        $paymentRequest = $builder->build();
        try {
            $result = $this->getApiClient()->createPayment($paymentRequest);
            if ($result) {
                if ($result->confirmation->type == 'redirect') {
                    header ('Location: ' . $result->confirmation->confirmationUrl);
                    exit;
                }
            }
        } catch (ApiException $e) {
            error_log('Yandex Kassa: Error order creation '. json_encode($e));
        }
        // Order Failed
        Link::redirect ('checkout', 'success');

    }



    /**
     * @return YandexClient
     */
    private function getApiClient()
    {
        if (!$this->apiClient) {
            $shopId          = $this->getOption ('ShopID');
            $shopPassword    = $this->getOption ('shopPassword');
            $this->apiClient = new YandexClient();
            $this->apiClient->setAuth($shopId, $shopPassword);
        }

        return $this->apiClient;
    }


    public function capturePayment($apiClient,  $payment_id)
    {
        $builder = CreateCaptureRequest::builder();
        $builder->setAmount($this->order->getTotalPrice ());
        $captureRequest = $builder->build();
        $payment = $apiClient->capturePayment($captureRequest,$payment_id);
        return $payment;
    }

    protected function getBuilder($order)
    {

        $builder = CreatePaymentRequest::builder()
            ->setAmount($this->order->getTotalPrice())
            ->setPaymentMethodData('')
            ->setCapture(true)
            ->setDescription('Оплата заказа №'.$this->order->getOrderId())
            ->setConfirmation(
                array(
                    'type'      => 'redirect',
                    'returnUrl' => Link::storeLink ('checkout', 'success', array ('provider' => $this->getName ()), false)
                )
            )
            ->setMetadata(array(
                'cms_name'       => 'motocms',
                'order_id' => $this->order->getOrderId(),
            ));
        return $builder;
    }


}
