<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com>
 **********************************************************************************************************************/

namespace MotoStore\Product\Repository;


use Doctrine\DBAL\LockMode;
use Doctrine\ORM\EntityNotFoundException;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\Query\Expr\Join;
use MotoStore\Content\Helper\Link;
use MotoStore\Product\Entity\Category;

/**
 * Class CategoryRepository
 * @package MotoStore\Product\Repository
 */
class CategoryRepository extends EntityRepository
{
    /**
     * Current store directory
     *
     * @var Category
     */
    private $category;
    /**
     * Categories Tree
     *
     * @var array
     */
    private $tree = array ();

    /**
     * @var array
     */
    private $breadcrumbs = array ();


    private $categories = array();

    /**
     * @param mixed $id
     * @param int $lockMode
     * @param null $lockVersion
     * @return null|object
     */
    public function find($id, $lockMode = LockMode::NONE, $lockVersion = null)
    {
        $entity =  parent::find($id, $lockMode, $lockVersion);

        $this->normalizeExternalResource ($entity);

        return $entity;
    }


    /**
     * Get categories tree
     * @return array
     */
    public function getTree ($show_btob = true)
    {
        if (!$this->tree)
        {
            $this->tree = $this->fetchTree ($show_btob);
        }
        return $this->tree;
    }

    /**
     * @return array
     */
    private function fetchTree ($show_btob = true)
    {
        return $this
            ->buildNestedTree ($this->getCategoryList ($show_btob));
    }

    /**
     * @return array
     */
    private function getCategoryList ($show_btob = true)
    {
        if (!$this->categories)
        {
            $query = $this->_em->createQueryBuilder ();

            $query
                ->select (array (
                    'c',
                    'l',
                    'i'
                ))
                ->from ('MotoStore\Product\Entity\Category', 'c')
                ->leftJoin ('c.locales', 'l')
                ->leftJoin ('c.image', 'i')
                ->addOrderBy ('c.parent_id', 'ASC')
                ->addOrderBy ('c.order', 'ASC');

            if(!$show_btob) {
                $query->where('c.btob != 1')->orWhere('c.btob is NULL');
            }


            $this->categories = $query
                ->getQuery ()
                ->getArrayResult ();
            /**
             * free query params
             */
            $query->getQuery ()->free ();
        }

        return $this->categories;
    }

    /**
     * @param array $categories
     * @return array
     */
    private function buildNestedTree ($categories)
    {
        $map = array ();
        $root = array ();
        for ($i = 0; $i < count ($categories); ++$i)
        {
            $map [$categories [$i] ['id']] = $i;
            $categories [$i] ['categories'] = array();
        }
        for ($i = 0; $i < count ($categories); ++$i)
        {
            $node = &$categories [$i];
            if ($node ['parent_id'] != 0)
            {
                if (isset($map [$node ['parent_id']]) ){
                    $categories [$map [$node ['parent_id']]] ['categories'][] = &$node;
                }
            }
            else
            {
                $root [] = &$node;
            }
        }
        return $root;
    }

    /**
     * Return current store directory
     * @param $uri
     * @return Category|null|object
     */
    public function getCategory ($uri)
    {
        if (!$this->category)
        {
            $this->category = $this->_em
                ->getRepository ('MotoStore\Product\Entity\Category')
                ->findOneBy (array (
                    'uri' => $uri
                ));

            $this->normalizeExternalResource ($this->category);
        }
        return $this->category;
    }

    /**
     * Return current store directory
     * @param $uri
     * @return Category|null|object
     */
    public function getCategoryById ($id)
    {
        if (!$this->category)
        {
            $this->category = $this->_em
                ->getRepository ('MotoStore\Product\Entity\Category')
                ->findOneBy (array (
                    'id' => $id
                ));

            $this->normalizeExternalResource ($this->category);
        }
        return $this->category;
    }

    /**
     * Return category breadcrumbs
     *
     * @param Category $category
     * @return array
     */
    public function getBreadcrumbs (Category $category)
    {
        $this->breadcrumbs [$category->getId ()] = array (
            'title' => $category->getLocaleByLanguageCode ('en')->getName (),
            'link'  => Link::storeLink (Link::URI_SECTION_CATEGORY, $category->getUri ())
        );

        if ($category->getParentId ())
        {
            $category = $this->find ($category->getParentId ());

            if ($category)
            {
                $this->getBreadcrumbs ($category);
            }
        }

        return array_reverse ($this->breadcrumbs, true);
    }

    /**
     * @param Category|null $entity
     */
    private function normalizeExternalResource (Category $entity = null)
    {
        if ($entity)
        {
            try
            {
                if ($entity->getImage ())
                {
                    $entity->getImage ()->getPath ();
                }
            }
            catch (EntityNotFoundException $e)
            {
                $entity->attachImage (null);
            }

        }
    }

    public function fetchCategoryProducts ($categoryID)
    {

        $qb = $this->_em->createQueryBuilder ();

        $query = $qb
            ->select (array (
                'i.path',
                'p.sku',
                'p.upc',
                'l.name',
                'r.product_id',
                'r.category_id',
                'r.position',
                'p.id'
            ))
            ->from ('MotoStore\Product\Entity\Product', 'p')
            ->leftJoin ('p.locales', 'l')
            ->leftJoin ('p.image', 'i')
            ->innerJoin('MotoStore\Product\Entity\Category\Relation', 'r', Join::WITH, 'p.id=r.product_id')
            ->where ($qb->expr ()->eq ('r.category_id', $categoryID))
            ->orderBy('r.position', 'ASC');


        return $query->getQuery()->getArrayResult ();
    }
}