<?php
/***********************************************************************************************************************
 * @author: <kolomiets.dev@gmail.com> 
 **********************************************************************************************************************/

namespace MotoStore\Update\Migration;

use MotoStore\Platform\Doctrine\Service\EntityManager;
use MotoStore\Settings\Entity\Option;
use MotoStore\Settings\Repository\OptionRepository;
use MotoStore\Update\Schema\AbstractSchema;
use MotoStore\Update\Version\Sample;
use vierbergenlars\SemVer\version;
use Moto;

class Migration
{
    /**
     * @var EntityManager
     */
    private  $entityManager;

    private $platformVersion;

    private $optionRepository;

    private $databaseVersion;

    private $migrationsSetting = array();

    private $globalConfig;

    private $installLockFilename;

    private $updateLockFilename;


    public function __construct (EntityManager $entityManager, OptionRepository $optionRepository, $platformVersion, $globalConfig = array())
    {
        $this->entityManager = $entityManager;
        $this->platformVersion = $platformVersion;
        $this->optionRepository = $optionRepository;
        $this->globalConfig = $globalConfig;
        $this->migrationsSetting =  Moto\Util::getFromArrayDeep($this->globalConfig,'migrations', array());
        $this->installLockFilename = Moto\Util::getFromArrayDeep($this->globalConfig,'lockFiles.install', MOTOSTORE_ROOT_DIR . '/data/system/install.lock');
        $this->updateLockFilename = Moto\Util::getFromArrayDeep($this->globalConfig,'lockFiles.update', MOTOSTORE_ROOT_DIR . '/data/system/auto-update.lock');
    }
    /**
     * Retrieve current database version
     *
     * @return string
     */
    private function getDatabaseVersion ()
    {
        if (!$this->databaseVersion)
        {
            if ($this->isInstalled ())
            {
                /**
                 * Temp fix for first update
                 */
                $firstUp = Moto\System::getAbsolutePath($this->updateLockFilename);

                $this->databaseVersion = $this->optionRepository
                    ->getOption (Option::OPTION_STORE_SETTINGS_SYSTEM, 'CURRENT_VERSION');

                if (in_array($this->databaseVersion, array ('1.1.3', '1.1.4')) && !file_exists ($firstUp))
                {
                    $this->databaseVersion = '1.1.2';
                }
            }
            else
            {
                $this->databaseVersion = '0.0.0';
            }
        }
        return $this->databaseVersion;
    }

    /**
     * Update current database version
     *
     * @param $version
     * @return string
     */
    private function updateDatabaseVersion ($version)
    {
        $this->optionRepository
            ->setOption (Option::OPTION_STORE_SETTINGS_SYSTEM, 'CURRENT_VERSION', $version);
    }

    /**
     * Check for autdate database
     * @return bool
     */
    private function isOutdate ()
    {
        return version::lt ($this->getDatabaseVersion (), $this->platformVersion);
    }

    /**
     * Migrate
     */
    public function migrate ()
    {
        if ($this->isOutdate ())
        {
            $this->clearAnnotationCache ();
            $this->run ();
            if (!$this->isInstalled())
            {
                $this->writeLockFile ();
            }
        }

    }

    private function run ()
    {
        foreach ($this->fetchMigrationsVersion () as $id)
        {
            $migrationClassName = $this->migrationsSetting [$id];
            /** @var AbstractSchema $migration */
            $migration = new $migrationClassName ($this->entityManager);
            $migration->update();
            $migration->import();
        }

        if (!$this->isInstalled ())
        {
            $sample = new Sample ($this->entityManager);
            $sample->update ();
            $sample->import ();
        }

        $this->updateDatabaseVersion ($this->platformVersion);
        if($this->getMigrationNumber ($this->getDatabaseVersion ()) < 115) {
            Moto\Util::filePutContents(Moto\System::getAbsolutePath($this->updateLockFilename),' 1013');;
        }
    }

    /**
     * Clear Annotation Cache
     */
    private function clearAnnotationCache ()
    {
        array_map ('unlink', glob ( Moto\System::getAbsolutePath(Moto\Util::getFromArrayDeep($this->globalConfig,'Doctrine.annotationDirectory', MOTOSTORE_ROOT_DIR . '/data/cache/annotation/')).'/*'));
    }

    /**
     * @return mixed
     */
    private function fetchMigrationsVersion ()
    {
        $currentVersion = $this->getMigrationNumber ($this->getDatabaseVersion ());
        if (in_array($currentVersion, array(1210,1211,1212,1213))){
            $currentVersion = 129;
        }
        $expectedVersion = $this->getMigrationNumber ($this->platformVersion);

        return array_filter (array_keys($this->migrationsSetting), function ($id) use ($currentVersion, $expectedVersion) {

            return $id > $currentVersion && $id <= $expectedVersion;
        });
    }

    /**
     * @param $version
     * @return int
     */
    private function getMigrationNumber ($version)
    {
        return (int) str_replace ('.', '', $version);
    }

    /**
     * Check if plugin installed
     * @return bool
     */
    protected function isInstalled ()
    {
        return file_exists (Moto\System::getAbsolutePath($this->installLockFilename));

    }

    /**
     * Writing Installation Lock FIle
     */
    private function writeLockFile ()
    {
        if (false ===  Moto\Util::filePutContents(Moto\System::getAbsolutePath($this->installLockFilename),' Dont remove this file'))
        {
            error_log ('install:error Failed to write install.lock');
        }
    }
}